/* This is a C app that can be used to launch a REXX script. It
 * captures all trace output (to the console window) to an error
 * log file. This includes any error message and traceback as a
 * result of some condition being raised.
 */

#ifdef WIN32
#include <windows.h>
#elif defined(OS2)
#include <os2.h>
#endif
#include <stdio.h>
#include <rexxsaa.h>

/* The handle to our error log file. If not yet open,
 * this will be -1.
 */
FILE *LogFile = (FILE *)-1;

/* Pointer to name of the error log file. If 0, then
 * no log file is desired.
 */
char *Filename = 0;

/* Here's my RXSIO Exit Handler. I use this to trap the
 * script's trace output and write it to a file.
 */

LONG APIENTRY MyRxsio(LONG exNum, LONG subfun, PEXIT pBlock)
{
	/* Determine which type of operation -- ie, is it a SAY output,
	 * is it a PULL input, or is it interactive TRACE input?
	 */
	switch (subfun)
	{
		/* It's the interpreter wanting us to display the next
		 * tracing line, or error message
		 */
		case RXSIOTRC:
		{
			RXSIOTRC_PARM * psiotrc;

			/* Redefine the PEXIT struct as what it really is
			 * -- a RXSIOTRC_PARM struct
			 */
			psiotrc = (RXSIOTRC_PARM *)pBlock;

			/* Write the line to our error log file. If the file isn't
			 * yet open, then open it as a new, empty file.
			 */
			if (Filename && (LogFile != (FILE *)-1 || (LogFile = fopen(Filename, "w")) != (FILE *)-1))
			{
				fputs(psiotrc->rxsio_string.strptr, LogFile);
				fputs("\n", LogFile);
			}
		}
	}


	/* Tell the interpreter to do what it would
	 * normally do.
	 */
	return(RXEXIT_NOT_HANDLED);
}

int main(int argc, char *argv[])
{
	RXSTRING	args[16];	/* MAX 16 args */
	RXSYSEXIT	exits[2];
	APIRET		err;

	/* Make sure we have a script name */
	if (argc < 2)
	{
		printf("USAGE: %s script_name [log_file] [script_args...]\n", argv[0]);
		return(-1);
	}

	/* Do we have a log file name? */
	if (argc > 2)
	{
		Filename = argv[2];
	}

	/* Stuff the arguments in our RXSTRING structs */
	for (err = 3; err < (APIRET)argc; err++)
	{
		args[err - 3].strptr = argv[err];
		args[err - 3].strlength = strlen(argv[err]);
	}

	/* Register an RXSIO Exit Handler to capture trace output
	 * from the interpreter/script
	 */
	if ((err = RexxRegisterExitExe("MyRxsio", (PFN)MyRxsio, NULL)))
	{
		printf("Can't register Exit handler!\n");
		return(-1);
	}

	/* Run the REXX Script named *argv[1] with my RXSIO Exit handler */
	exits[0].sysexit_name = "MyRxsio";
	exits[0].sysexit_code = RXSIO;
	exits[1].sysexit_code = RXENDLST;
	err = RexxStart(argc - 3, &args[0], argv[1], 0, 0, RXSUBROUTINE, exits, 0, 0);

	/* Deregister my RXSIO Exit Handler */
	RexxDeregisterExit("MyRxsio", NULL);

	/* Close any open log file */
	if (LogFile != (FILE *)-1) fclose(LogFile);

	return(0);
}