/* An example windowed app (for MS Windows) which installs an RXSIO Exit Handler to
 * trap and process the REXX script's input/output. One caveat: If the script uses
 * CHARIN()/CHAROUT()/LINEIN()/LINEOUT() to read/write to stdin/stdout, then we're
 * in trouble without a console window open. For this reason, it would also be good
 * to utilize Reginald's CONSOLE OPTION. (But that is proprietary to Reginald, so
 * if you wish to support other interpreters, realize that you have a potential
 * problem with them).
 */

#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#include <signal.h>
#include <io.h>
#include <stdio.h>
#include "..\rexxsaa.h"
#include "resource.h"

/* Define this if you intend to support only Reginald */
/* #define REGINALD_EXTENSIONS */





struct RXFETCHLINE
{
	RXSTRING	*retstr;
	char		*title;
};






/* The Dialog procedure for my "Enter a line" dialog. The operating system
 * calls this when it pass my dialog a message.
 *
 * I use this dialog to get a line of input from the user, in order to
 * return it to the REXX interpreter in MyRxsio() for interactive trace
 * input or input for the PULL instruction.
 */

UINT APIENTRY EnterALineProc(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch (msg)
	{
		/* Dialog is opening */
		case WM_INITDIALOG:
		{
			HWND				hwndChild;
			struct RXFETCHLINE *fetch;

			/* Get the RXFETCHLINE struct passed via DialogBoxParam() */
			fetch = (struct RXFETCHLINE *)lParam;

			/* Set the window title */
			SetWindowText(hDlg, fetch->title);

			/* Allow him to enter a line upto 1024 characters. This is arbitrary */
			if ((hwndChild = GetDlgItem(hDlg, IDC_MY_LINE)))
			{
				SendMessage(hwndChild, EM_SETLIMITTEXT, 1024, 0);

				/* Save the passed pointer to our RXFETCHLINE struct in the
				 * Dialog's DWL_USER field so that we can retrieve it later
				 */
				SetWindowLong(hDlg, DWL_USER, (LPARAM)fetch);

				/* Success */
				return(TRUE);
			}

			/* Error */
			break;
		}

		case WM_COMMAND:
		{
			/* We care about only button clicks */
			if (HIWORD(wParam) == BN_CLICKED)
			{
				switch (LOWORD(wParam))
				{
					/* User selected Ok */
					case IDOK:
					{
						struct RXFETCHLINE *fetch;

						/* Retrieve the pointer to our RXFETCHLINE struct that we
						 * saved when it was passed to DialogBoxParam()
						 */
						fetch = (struct RXFETCHLINE *)GetWindowLong(hDlg, DWL_USER);

						/* Copy the text in the edit box to the supplied RXSTRING's data
						 * buffer which will be returned to the interpreter by MyRxsio().
						 * NOTE: This doesn't have to be nul-terminated although
						 * GetDlgItemText() will do that
						 */
						GetDlgItemText(hDlg, IDC_MY_LINE, fetch->retstr->strptr, fetch->retstr->strlength);

						/* Set the RXSTRING's strlength */
						fetch->retstr->strlength = strlen(fetch->retstr->strptr);

						/* Close the Dialog and return 1 */
						EndDialog(hDlg, 1);

						break;
					}

					/* User selected Cancel */
					case IDCANCEL:
					{
						/* Close the Dialog and return 0 */
						EndDialog(hDlg, 0);

						break;
					}
				}
			}

			break;
		}

		/* ================== User wants to close window ==================== */
		case WM_CLOSE:
		{
			/* Close the Dialog (assuming cancel) */
			EndDialog(hDlg, 0);
			break;
		}
    }

	// Let the OS handle it
	return(FALSE);
}





/* Here's my RXSIO Exit Handler. I use this to trap the interpreter/script's
 * input/output. So, I don't need a console window open. I can redirect
 * everything to/from windows/dialogs created with CreateWindowEx(),
 * DialogBox(), etc.
 */

LONG APIENTRY MyRxsio(LONG exNum, LONG subfun, PEXIT pBlock)
{
	struct RXFETCHLINE	fetch;

	/* Determine which type of operation -- ie, is it a SAY output, is it
	 * a PULL input, or is it interactive TRACE input?
	 */
	switch (subfun)
	{
		/* It's the SAY instruction wanting us to output a line */
		case RXSIOSAY:
		{
			RXSIOSAY_PARM * psiosay;

			/* Redefine the PEXIT struct as what it really is -- a RXSIOSAY_PARM struct */
			psiosay = (RXSIOSAY_PARM *)pBlock;

			/* Just print the line in a message box */
			MessageBox(0, psiosay->rxsio_string.strptr, "SAY", MB_OK);

			break;
		}

		/* It's the interpreter wanting us to display the next tracing line, or error message */
		case RXSIOTRC:
		{
			RXSIOTRC_PARM * psiotrc;

			/* For trace output, we really should do something like add the line to
			 * the end of an edit control which is collecting all of the lines so far.
			 * Trace output entails lots of lines displayed and we don't really want
			 * a message box popping up with each individual line. Plus, it's more
			 * meaningful for the user to see several lines at once. The problem is
			 * that the interpreter also outputs error messages through an RXSIOTRC
			 * operation. So we don't know which are really trace lines and which
			 * are error messages. This was really not a good design for REXX. It
			 * should have a separate operation for error message display since it's
			 * important to distinguish those for a user
			 */

			/* Redefine the PEXIT struct as what it really is -- a RXSIOTRC_PARM struct */
			psiotrc = (RXSIOTRC_PARM *)pBlock;

			/* Just print the line in a message box */
			MessageBox(0, psiotrc->rxsio_string.strptr, "Trace -- Error", MB_OK);

			break;
		}

		/* It's the PULL instruction wanting us to get a line from the user */
		case RXSIOTRD:
		{
			RXSIOTRD_PARM		*psiotrd;
			int					code;

			/* Redefine the PEXIT struct as what it really is -- a RXSIOTRD_PARM struct */
			psiotrd = (RXSIOTRD_PARM *)pBlock;

			/* Assume no input returned */
			psiotrd->rxsiotrd_retc.strlength = 0;

			/* Set up our RXFETCHLINE struct to pass appropriate data to EnterALineProc() */
			fetch.title = "Enter a line for PULL:";
			fetch.retstr = &psiotrd->rxsiotrd_retc;

			/* Get user's input via a dialog box with an edit control and Ok/Cancel buttons */
fetch_it:	if ((code = DialogBoxParam(GetModuleHandle(0), MAKEINTRESOURCE(IDD_ENTER_A_LINE), 0, EnterALineProc, (LPARAM)&fetch)) == -1)
			{
				/* An error. Tell the interpreter to raise the SYNTAX condition
				 * (which will abort the script if it doesn't trap SYNTAX) and
				 * display a generic error message associated with ANSI REXX
				 * General error number RXERR_SYSTEM_FAILURE.
				 *
				 * NOTE: If the script doesn't trap SYNTAX, us telling the
				 * interpreter to raise SYNTAX will cause the interpreter to
				 * eventually call MyRxsio() again (after MyRxsio() returns here)
				 * with a RXSIOTRC operation to display an error message
				 * associated with RXERR_SYSTEM_FAILURE
				 */
				return(RXEXIT_RAISE_ERROR);
			}

			/* Did he click upon the Cancel button? */
			if (!code)
			{
				/* Yes. Let's raise a HALT condition in the interpreter. Reginald
				 * offers a proprietary function, RexxRaiseCondition(), for
				 * this purpose which allows us a lot more flexibility in this
				 * regard than the standard API RexxSetHalt(). Besides raising
				 * the HALT condition, we could alternately choose to raise the
				 * SYNTAX condition with one of the ANSI REXX General Error
				 * numbers. You'll find those numbers defined in REXXSAA.H
				 * under the "EXTERNAL FUNCTION HANDLERS" section (for example,
				 * RXERR_STORAGE_EXHAUSTED). That allows us to abort the script's
				 * execution (if it doesn't trap SYNTAX) with a specific,
				 * meaningful error message. Contrast that to the only option
				 * that other interpreters allow for raising SYNTAX -- returning
				 * RXEXIT_RAISE_ERROR -- which always reports a ANSI GE error of
				 * RXERR_SYSTEM_FAILURE. (ie, No choice of error message there)
				 *
				 * Or, we can raise the HALT condition which will also abort
				 * the script (if it doesn't trap HALT), but specify any
				 * signal number. Contrast that to the only option that other
				 * interpreters allow for raising HALT -- calling RexxSetHalt()
				 * -- which always specifies a signal of "SIGINT".
				 *
				 * In conclusion, Reginald's RexxRaiseCondition() offers the ability
				 * to report a much wider range of error messages/numbers to the
				 * script, and thereby report much more detailed, accurate error
				 * feedback to the user.
				 *
				 * NOTE: If the script doesn't trap HALT, our raising HALT
				 * condition below will cause the interpreter to eventually call
				 * MyRxsio() again (after MyRxsio() returns here) with a RXSIOTRC
				 * operation to display an error message that the script has been
				 * aborted
				 */
#ifdef USE_REGINALD_EXTENSIONS
				RexxRaiseCondition(REXXRAISE_HALT, SIGINT, 0, 0);
#else
				RexxSetHalt(GetCurrentProcess(), GetCurrentThread());
#endif
				/* Since we just raised the condition ourselves, we return
				 * RXEXIT_HANDLED so that the interpreter doesn't also
				 * raise a condition (RXEXIT_RAISE_ERROR), nor attempt
				 * to handle inputting a line itself (RXEXIT_NOT_HANDLED)
				 */
			}

			/* If he clicked upon Ok, then the string has already been copied to the
			 * supplied RXSTRING and its strlength set in EnterALineProc(). All that
			 * is left is to return RXEXIT_HANDLED
			 */

			break;
		}

		/* It's interactive trace wanting us to get a line from the user */
		case RXSIODTR:
		{
			RXSIODTR_PARM		*psiodtr;

			/* Redefine the PEXIT struct as what it really is -- a RXSIODTR_PARM struct */
			psiodtr = (RXSIODTR_PARM *)pBlock;

			/* Assume no input returned */
			psiodtr->rxsiodtr_retc.strlength = 0;

			/* Set up our RXFETCHLINE struct to pass appropriate data to EnterALineProc() */
			fetch.title = "Enter a tracing instruction:";
			fetch.retstr = &psiodtr->rxsiodtr_retc;

			goto fetch_it;
		}

		default:
		{
			/* Hmmmmm. Must be some new REXX thing implemented after our program was written,
			 * Tell the interpreter to take care of this operation since we don't know what
			 * to do here
			 */
			return(RXEXIT_NOT_HANDLED);
		}
	}

	/* We successfully handled this operation. Let the interpreter
	 * continue running the script with the next operation
	 */
	return(RXEXIT_HANDLED);
}





int WINAPI WinMain(HINSTANCE  hInstance, HINSTANCE  hPrevInstance, LPSTR  lpCmdLine, int  nShowCmd)
{
	RXSYSEXIT	exits[2];
	APIRET		err;
	char		buffer[40];

	/* Register an RXSIO Exit Handler to receive any output from the
	 * interpreter/script, and to feed the script any input
	 */
	if ((err = RexxRegisterExitExe("MyRxsio", (PFN)MyRxsio, NULL)))
	{
		MessageBox(0, "Failed to register MyRxsio", "ERROR", MB_OK);
		return(-1);
	}

	/* Run the REXX Script named "test.rex" in the current directory
	 * with my RXSIO Exit handler
	 */
	exits[0].sysexit_name = "MyRxsio";
	exits[0].sysexit_code = RXSIO;
	exits[1].sysexit_code = RXENDLST;
	err = RexxStart(0, 0, "test.rex", 0, 0, RXSUBROUTINE, exits, 0, 0);

#ifdef REGINALD_EXTENSIONS
	/* If Reginald's RexxStart() returned an error number < RX_START_SPECIAL,
	 * then we have to display an error message ourselves, since we don't
	 * have an open console. Otherwise, RexxStart() takes care of displaying
	 * all error messages.
	 */
	if (err && err < RX_START_SPECIAL)
	{
		sprintf(&buffer[0], "RexxStart() error: %u", err);
		MessageBox(0, &buffer[0], "ERROR", MB_OK|MB_ICONSTOP);
	}
#else
	/* For other interpreters, it's hit or miss whether RexxStart()
	 * displays an error message, so we do it here regardless.
	 */
	if (err)
	{
		sprintf(&buffer[0], "RexxStart() error: %u", err);
		MessageBox(0, &buffer[0], "ERROR", MB_OK|MB_ICONSTOP);
	}
#endif

	/* Deregister my RXSIO Exit Handler */
	if ((err = RexxDeregisterExit("MyRxsio", NULL)))
	{
		MessageBox(0, "Failed to deregister MyRxsio", "ERROR", MB_OK);
		return(-1);
	}

	return(0);
}