/* An example windowed app (for MS Windows) which opens its own console window
   for the REXX script's input/output
 */

#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#include <signal.h>
#include <io.h>
#include "..\rexxsaa.h"

/* Define this if you intend to support only Reginald */
/* #define REGINALD_EXTENSIONS */



const char	AnyKey[] = "\r\nPress any key to close the window...\r\n";
const char	ErrorStr[] = "REXX Launch Error";
const char	BadConsole[] = "Can't allocate console!";
const char	BadTermSetup[] = "Can't setup script termination handling!";




/* *********************** ctrlCHandler() ***********************
 * Called by Windows whenever the user presses a certain key combo
 * (such as CTRL-C or BREAK), or when the OS needs to inform us of
 * some action (such as shutdown).
 *
 * dwCtrlType =	Action type as passed to us by Windows.
 */

BOOL WINAPI ctrlCHandler(DWORD dwCtrlType)
{
	switch (dwCtrlType)
	{
		case CTRL_C_EVENT:
		{
			/* Tell interpreter to signal CTRL-C abort */
#ifdef REGINALD_EXTENSIONS
			RexxRaiseCondition(REXXRAISE_HALT, SIGINT, 0, 0);
#else
			RexxSetHalt(GetCurrentProcess(), GetCurrentThread());
#endif
			return(TRUE);
		}

		case CTRL_CLOSE_EVENT:
		{
			/* Tell interpreter to signal program termination */
#ifdef REGINALD_EXTENSIONS
			RexxRaiseCondition(REXXRAISE_HALT, SIGTERM, 0, 0);
#else
			RexxSetHalt(GetCurrentProcess(), GetCurrentThread());
#endif
			return(TRUE);
		}

		case CTRL_LOGOFF_EVENT:
		case CTRL_SHUTDOWN_EVENT:
		{
			/* Tell interpreter to signal program termination */
#ifdef REGINALD_EXTENSIONS
			RexxRaiseCondition(REXXRAISE_HALT, SIGTERM, 0, 0);
#else
			RexxSetHalt(GetCurrentProcess(), GetCurrentThread());
#endif
		}
	}

	/* Do default action for the remainder of these types */
	return(FALSE);
}





int WINAPI WinMain(HINSTANCE  hInstance, HINSTANCE  hPrevInstance, LPSTR  lpCmdLine, int  nShowCmd)
{
	HANDLE	fh;
	APIRET	err;

	/* Allocate a console in which to run the REXX script */
	if (AllocConsole())
	{
		/* Set input mode to return every character */
		fh = GetStdHandle(STD_INPUT_HANDLE);
		SetConsoleMode(fh, ENABLE_ECHO_INPUT|ENABLE_PROCESSED_INPUT);

		/* Set up handling of system termination request and CTRL-C */
		if (!SetConsoleCtrlHandler(ctrlCHandler, 1))
		{
			FreeConsole();
			MessageBox(0, &BadTermSetup[0], &ErrorStr[0], MB_OK);
			return(-1);
		}

		/* Run the REXX Script named "test.rex" in the current directory */
		err = RexxStart(0, 0, "test.rex", 0, 0, RXSUBROUTINE, 0, 0, 0);

		/* Was there an error? If so let's leave the console window open (so that
			the user can read the error message) until he presses a key
		*/
		if (err)
		{
			DWORD  size;
			char   buffer;

			fh = GetStdHandle(STD_OUTPUT_HANDLE);

			/* Tell user to press any key to close, and wait for him to do so */
			WriteFile(fh, &AnyKey[0], sizeof(AnyKey) - 1, &size, 0);
			fh = GetStdHandle(STD_INPUT_HANDLE);
			ReadFile(fh, &buffer, 1, &size, 0);
		}

		/* Close the console */
		FreeConsole();
	}
	else
		MessageBox(0, &BadTermSetup[0], &ErrorStr[0], MB_OK);

	return(0);
}