/* An example that registers two External Functions, MyAdd() and DisplayArgs() */

#ifdef WIN32
#include <windows.h>
#endif
#include <stdio.h>
#define RX_STRONGTYPING
#include "..\rexxsaa.h"





/* Here's one of our REXX-callable functions that we install via
   RexxRegisterFunctionExe(). This one expects two numeric args
   from the script. It then adds the args and returns the sum
   to the script.
 */

APIRET APIENTRY MyAdd(CONST CHAR *name, ULONG numargs, RXSTRING args[], CONST UCHAR *queuename, RXSTRING *retstr)
{
	char	*stopptr;
	long	val1, val2;

	/* Check that there are 2, and only 2, args */
	if (numargs != 2 ||

		/* Make sure that both the args are there (ie, not omitted) */
		!args[0].strptr || !args[0].strlength || !args[1].strptr || !args[1].strlength)
	{
		/* Hey! He didn't supply at least 2 args. That's an error to me */

		/* One nice feature of Reginald is that, if you return RXERR_EXTFUNC_ERR
		 * (instead of simply RXERR_INCORRECT_CALL), you can copy an error
		 * message to 'retstr' and Reginald will report that to the script
		 * (or display the error message to the user).
		 *
		 * But even better, if you use one of the RXERR_ numbers that correspond
		 * to the ANSI GE numbers, or use the error numbers that utilize the
		 * MAKEFUNCARG() macro, you don't even need to supply an error message
		 * -- Reginald supplies one for you. Plus Reginald's online help can
		 * present a page on that specific error.
		 *
		 * This is a lot more informative than just reporting an "Incorrect call
		 * to function" as with other interpreters. But, only Reginald supports
		 * this. Although the following code should do nothing special on other
		 * properly written interpreters (ie, it will result in only that
		 * "Incorrect call to function" error message), it will allow Reginald
		 * to report more detailed errors for your function. Reginald will also
		 * report the name of our function, so we don't need to embed that in
		 * our error message.
		 */

		 /* Ok, did he omit both args? */
		if (!numargs || (!args[0].strptr && !args[1].strptr))
		{
			strcpy(retstr->strptr, "Arguments 1 and 2 can't be omitted");

			/* Note that we didn't have to nul-terminate the error message, but
			 * it was easier to do so above. Nevertheless, don't count that nul
			 */
			retstr->strlength = strlen(retstr->strptr);

			/* Tell Reginald to use the error message we're returning in 'retstr' */
			return(RXERR_EXTFUNC_ERR);

			/* Instead of doing the above, we could have simply done:
			 * return(RXERR_TOOFEWARGS | MAKEFUNCARG(2));
			 * But I wanted to show you how to return your own error message.
			 */
		}

		/* For the rest of these errors, we don't need to return a string.
		 * Reginald doesn't care about setting retstr to 0 if you return an
		 * error number. But other interpreters may
		 */
#ifndef USING_REGINALD
		retstr->strlength = 0;
#endif

		/* Did he pass in more than 2 args? */
		if (numargs > 2)
		{
			return(RXERR_TOOMANYARGS | MAKEFUNCARG(2));
		}
		 /* Ok, did he omit the first arg? */
		if (!args[0].strptr)
		{
			return(RXERR_ARGMISSING | MAKEFUNCARG(1));
		}
		 /* Ok, did he omit the second arg? */
		if (numargs < 2 || !args[1].strptr)
		{
			return(RXERR_ARGMISSING | MAKEFUNCARG(2));
		}
		 /* Ok, did he supply an empty string for the first arg? */
		if (!args[0].strlength)
		{
			return(RXERR_ARGNOTNUL | MAKEFUNCARG(1));
		}
		 /* Ok, did he supply an empty string for the second arg? */
		if (!args[1].strlength)
		{
			return(RXERR_ARGNOTNUL | MAKEFUNCARG(2));
		}

		/* No specific error message */
		return(RXERR_INCORRECT_CALL);
	}

	/* Remember that REXX passes numbers as numeric strings. We have
	   to convert each to a binary value in order to do the addition.
	   We also perform a check to make sure that the script passed in
	   numeric args (ie, there weren't any characters in the arg which
	   weren't valid in a numeric arg)
	 */
	val1 = strtol(args[0].strptr, &stopptr, 10);
	if ((stopptr - args[0].strptr) != args[0].strlength)
	{
#ifndef USING_REGINALD
		retstr->strlength = 0;
#endif
		return(RXERR_ARGWHOLENUM | MAKEFUNCARG(1));
	}

	val2 = strtol(args[1].strptr, &stopptr, 10);
	if ((stopptr - args[1].strptr) != args[1].strlength)
	{
#ifndef USING_REGINALD
		retstr->strlength = 0;
#endif
		return(RXERR_ARGWHOLENUM | MAKEFUNCARG(2));
	}

	/* Perform the addition */
	val2 += val1;

	/* Now we must return the sum as a numeric string to REXX */
	sprintf(retstr->strptr, "%d", val2);
	retstr->strlength = strlen(retstr->strptr);

	/* Return successfully */
	return(RXFUNC_OK);
}





/* Here's another of our REXX-callable functions that we install via
   RexxRegisterFunctionExe(). This one expects any number of args
   from the script. It then displays those args.
 */

APIRET APIENTRY DisplayArgs(CONST CHAR *name, ULONG numargs, RXSTRING args[], CONST UCHAR *queuename, RXSTRING *retstr)
{
	ULONG i;

	for (i = 0; i < numargs; i++)
	{
		if (args[i].strptr)
			printf("Arg #%ld = %s\n", i + 1, args[i].strptr);
		else
			printf("Arg #%ld is omitted\n", i + 1);
	}

	/* We don't return any value */
	retstr->strlength = 0;

	/* Return successfully */
	return(RXFUNC_OK);
}





int main(int argc, char **argv)
{
	APIRET	err;
	char	chr;

	/* Register our function called "MyAdd() */
	err = RexxRegisterFunctionExe("MyAdd", MyAdd);
	if (err)
	{
		printf("Error #%ld registering MyAdd()\n", err);
	}
	else
	{
		/* Register our function called "DisplayArgs() */
		err = RexxRegisterFunctionExe("DisplayArgs", DisplayArgs);
		if (err)
		{
			printf("Error #%ld registering DisplayArgs()\n", err);
		}
		else
		{
			/* Run the REXX Script named "test.rex" in the current directory */
			err = RexxStart(0, 0,
							"test.rex",    /* name of REXX script */
							0, 0, RXSUBROUTINE, 0, 0, 0);

			/* Was there an error running the script? */
			if (err)
			{
				/* Yes. Print the error number except if using Reginald. Reginald's RexxStart()
				   always displays the error for you. Other interpreters are inconsistent.
				   Sometimes they do and sometimes they don't
				*/
#ifndef USING_REGINALD
				printf("RexxStart() error: %d\n", err);
#endif
			}

			/* Deregister "DisplayArgs() */
			err = RexxDeregisterFunction((CONST CHAR *)"DisplayArgs");
			if (err)
			{
				printf("Error #%ld deregistering DisplayArgs()\n", err);
			}
		}

		/* Deregister "MyAdd() */
		err = RexxDeregisterFunction((CONST CHAR *)"MyAdd");
		if (err)
		{
			printf("Error #%ld deregistering MyAdd()\n", err);
		}
	}

	/* Wait for user to press ENTER before ending */
	printf("Press ENTER to end...\n");
	scanf("%c", &chr);

	return(0);
}
