/* An example windowed app which neither installs an RXSIO Exit Handler nor opens its
 * own console window. Rather, it uses Reginald's proprietary CONSOLE OPTION. This
 * means that Reginald will open a console for the script, if necessary.
 *
 * This is a version of Script10 which uses dynamic linking to Reginald.
 */

#ifdef WIN32
#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#endif
#include "..\rexxsaa.h"





/* Here is my OPTIONS string passed to RexxSetOptions(). I'm setting the
 * MSGBOX, NULTERM, and CONSOLE options. The order of the items isn't
 * important, but they must be upper case. Also note that the string
 * doesn't need to be nul-terminated.
 */

CHAR MyOptions[] = {'C','O','N','S','O','L','E',' ','N','U','L','T','E','R','M',' ','M','S','G','B','O','X'};






/* Here's our program entry point. It is defined for MS Windows. A C compiler for another
 * operating system will likely require you to declare it with some other name, and with
 * other args passed to it.
 */

int WINAPI WinMain(HINSTANCE  hInstance, HINSTANCE  hPrevInstance, LPSTR lpCmdLine, int nShowCmd)
{
	RXSTRING			retstr;
	APIRET				err;
	RexxSetOptionsPtr	*RexxSetOptionsA;
	RexxFreeMemoryPtr	*RexxFreeMemoryA;
	RexxStartPtr		*RexxStartA;
	HMODULE				RexxHandle;

	/* Try to open Reginald's interpreter DLL */
	if ((RexxHandle = LoadLibrary("REGINALD")))
	{
		/* Ok, Reginald's interpreter DLL loaded */

		/* Get a pointer to the RexxSetOptions() API */
		if (!(RexxSetOptionsA = (RexxSetOptionsPtr *)GetProcAddress((HINSTANCE)RexxHandle, "RexxSetOptions")) ||
			
			/* Get a pointer to the RexxFreeMemory() API */
			!(RexxFreeMemoryA = (RexxFreeMemoryPtr *)GetProcAddress((HINSTANCE)RexxHandle, "RexxFreeMemory")) ||
			
			/* Get a pointer to the RexxStart() API */
			!(RexxStartA = (RexxStartPtr *)GetProcAddress((HINSTANCE)RexxHandle, "RexxStart")))
		{
			MessageBox(0, "Can't load REXX API", "REXX Error", MB_OK|MB_ICONSTOP);
			FreeLibrary(RexxHandle);
			return(-3);
		}
	}
	else
	{
		MessageBox(0, "Can't find Reginald REXX interpreter", "REXX Error", MB_OK|MB_ICONSTOP);
		return(-2);
	}

	/* Set the CONSOLE, MSGBOX, and NULTERM options */
	if ((*RexxSetOptionsA)(&MyOptions[0], sizeof(MyOptions)))
	{
		MessageBox(0, "Can't set REXX options", "REXX Error", MB_OK|MB_ICONSTOP);
		FreeLibrary(RexxHandle);
		return(-1);
	}

	/* Pass an RXSTRING to get any possible return value from the script. Let the
	 * interpreter allocate the data buffer
	 */
	retstr.strptr = 0;

	/* Run the REXX Script named "test.rex" in the current directory */
	err = (*RexxStartA)(0, 0, "test.rex", 0, 0, RXSUBROUTINE, 0, 0, &retstr);

	/* NOTE: If RexxStart() returned an error, it has already displayed
	 * an error message (in the console window, or in a message box if the
	 * MSGBOX OPTION is set). So we don't need to do that.
	 */

	/* If success, let's check if the script returned anything. We have to
	 * free that if it did.
	 */
	if (!err)
	{
		/* Did the script return some value? */
		if (retstr.strptr && retstr.strlength)
		{
			/* Print out any value returned by the script. NOTE: NULTERM
			 * option nul-terminates the data buffer. Of course, there could
			 * be embedded nul bytes in the data itself. We aren't checking
			 * for that here, since it isn't important to us
			 */
			MessageBox(0, retstr.strptr, "Script returned:", MB_OK);

			/* Free the data buffer */
			(*RexxFreeMemoryA)(retstr.strptr);
		}

		/* The script didn't return a value */
		else
		{
			MessageBox(0, "", "Script did not return a value", MB_OK);
		}
	}

	/* Close Reginald now that we're done with it */
	FreeLibrary(RexxHandle);

	/* All done! Easy, huh? */
	return(0);
}
