/* An example that registers an External Function that fetches the value of
 * a variable in the REXX script
 */

#ifdef WIN32
#include <windows.h>
#endif
#include <stdio.h>
#define RX_STRONGTYPING
#include "..\rexxsaa.h"




/* Here's our REXX-callable function that we install via
   RexxRegisterFunctionExe(). This fetches the value of a
   REXX variable (in the script) named "MyVar" and displays
   it.
 */

APIRET APIENTRY GetMyVar(PUCHAR name, ULONG numargs, RXSTRING args[], PSZ queuename, PRXSTRING retstr)
{
	SHVBLOCK block;
	APIRET   err;

	/* We don't return a value */
	retstr->strlength = 0;

	/* Let's fetch the value of MyVar */

	/* We're using only 1 SHVBLOCK (ie, 1 operation) */
	block.shvnext = 0;

	/* The variable name is "MyVar". Doesn't
		have to be nul-terminated, but it's easier */
	block.shvname.strptr = "MyVar";
	block.shvname.strlength = strlen(block.shvname.strptr);

	/* Let the interpreter provide the data buffer for the value */
	block.shvvalue.strptr = 0;
	block.shvvalue.strlength = 0;

	/* The operation is to fetch the variable */
	block.shvcode = RXSHV_FETCH;

	/* Do that operation */
	err = RexxVariablePool(&block);

	/* An error? Could even be that the variable doesn't exist (RXSHV_NEW),
	   in which case the value returned is the same as the variable name,
	   with any tail substitutions */
	if (err)
	{
		/* It's our responsibility to free this RXSTRING's data buffer,
			but first check if RexxVariablePool() allocated anything */
		if (block.shvvalue.strptr)
		{
			/* Was it just created anew? Perhaps you may not consider this an error */
			if (err & RXSHV_NEW)
			{
				/* Print the value */
				printf("MyVar didn't exist. Now it does. Actual name with tail substitution = ");

				/* It's not necessarily nul-terminated, so we can't treat
				   it as a C string, and pass it directly to printf() */
				for (err = 0; err < block.shvvalue.strlength; err++)
				{
					printf("%c", block.shvvalue.strptr[err]);
				}
				printf("\n");
			}

			RexxFreeMemory(block.shvvalue.strptr);
		}

		return(RXERR_INTERPRETER_FAILURE);
	}

	/* Print the value */
	printf("MyVar = ");

	/* It's not necessarily nul-terminated, so we can't treat
	   it as a C string, and pass it directly to printf() */
	for (err = 0; err < block.shvvalue.strlength; err++)
	{
		printf("%c", block.shvvalue.strptr[err]);
	}
	printf("\n");

	/* It's our responsibility to free this RXSTRING's data buffer */
	RexxFreeMemory(block.shvvalue.strptr);

	/* Return successfully */
	return(RXFUNC_OK);
}





int main(int argc, char **argv)
{
	APIRET	err;
	char	chr;

	/* Register our function called "GetMyVar() */
	err = RexxRegisterFunctionExe("GetMyVar", SetMyVar);
	if (err)
	{
		printf("Error #%ld registering GetMyVar()\n", err);
	}
	else
	{
		/* Run the REXX Script named "test.rex" in the current directory */
		err = RexxStart(0, 0, "test.rex", 0, 0, RXSUBROUTINE, 0, 0, 0);

		/* Was there an error running the script? */
		if (err)
		{
			/* Yes. Print the error number except if using Reginald. Reginald's RexxStart()
			   always displays the error for you. Other interpreters are inconsistent.
			   Sometimes they do and sometimes they don't
			*/
#ifndef USING_REGINALD
			printf("RexxStart() error: %d\n", err);
#endif
		}

		/* Deregister "GetMyVar() */
		err = RexxDeregisterFunction("GetMyVar");
		if (err)
		{
			printf("Error #%ld deregistering GetMyVar()\n", err);
		}
	}

	/* Wait for user to press ENTER before ending */
	printf("Press ENTER to end...\n");
	scanf("%c", &chr);

	return(0);
}
