/* An example windowed app which neither installs an RXSIO Exit Handler nor opens its
 * own console window. Rather, it uses Reginald's proprietary CONSOLE OPTION. This
 * means that Reginald will open a console for the script, if necessary.
 *
 * What is the advantage/use of this? Well first, a windowed app wouldn't *want* to
 * open a console unless it needed to. Windowed apps instead use OS functions to create
 * non-console (ie, graphical) windows for their user interface -- not text-based
 * console windows. For example, MS Windows uses CreateWindow(), DialogBox(), etc.
 * So, a windowed app wouldn't have a console unless it needed it for the purposes
 * of REXX.
 *
 * Furthermore, let's say that we want to register our own External Function
 * that lets the script display text to one of our windows (created via CreateWindow(),
 * DialogBox(), etc). We also want to register another External Function that
 * lets the script get a line of text from the user via another dialog box with
 * window controls such as listboxes, edit boxes, etc. In this way, the script can use
 * these special functions instead of SAY, PULL, etc, and the script can create a
 * nice graphical user interface too (instead of throwing up an ugly, text-based
 * console window). In this case, the script never needs an open console (since it
 * never does SAY, PULL, etc), so why should we even have it around?
 *
 * But let's say that we also want to be able to run scripts that don't take
 * advantage of these special functions, and do use SAY, PULL, etc. In that case, a
 * console window needs to be open only for those scripts.
 *
 * Sure, we could add an RXSIO Exit Handler that checks whether a console is
 * open, and open one if not. After all, the RXSIO Exit Handler is called only when
 * a line of text needs to be displayed to the console (as a result of a SAY
 * instruction, or tracing enabled), or a line of text needs to be input from the
 * user (as a result of PULL instruction, or interactive tracing). That would
 * theoretically be compatible with all interpreters (but in practice is not -- some
 * interpreters don't bother utilizing a console that is opened after the call to
 * RexxStart(). Some interpreters don't even recognize a console that is opened with
 * Windows AllocConsole()). But another problem is if the script uses
 * CHARIN()/LINEIN()/CHAROUT()/LINEOUT() to specifically output to stdin/stdout. This
 * bypasses the RXSIO Exit Handler, so we would never know that we needed a console
 * window open. Also, with an RXSIO Exit Handler, we need to worry about missing any
 * RexxStart() errors as a result of not opening a console window on our own before
 * calling RexxStart().
 *
 * Reginald's CONSOLE OPTION makes this all easy and foolproof. Just call Reginald's
 * proprietary RexxSetOptions() API to set the CONSOLE OPTION. Reginald takes care
 * of opening/closing a console window when needed.
 *
 * Furthermore, we also set Reginald's MSGBOX OPTION. With this option, Reginald
 * displays an error message in a pop-up box with a "Help" button that the user
 * can utilize to bring up a help page on that error. (Without this option, error
 * messages are displayed to the console window, as normal). This includes any
 * errors as a result of the script not trapping SYNTAX or HALT. This gives our
 * program a nice extra touch without any work on our part.
 *
 * Also, Reginald's NULTERM OPTION is enabled. This nul-terminates any data buffer
 * returned by the script from RexxStart(). This is useful when using any operating
 * system or C library functions that expect nul-terminated strings (such as MS Windows'
 * MessageBox()).
 *
 * In conclusion, Reginald's CONSOLE option allows a windowed app to run a REXX
 * script without worrying about opening any console, nor needing to install any
 * RXSIO Exit Handler. Reginald's NULTERM option makes it easier for a C app to deal
 * with return values. Reginald's MSGBOX option presents a "helpful" message box for
 * error messages.
 *
 * Look at the below code, and you'll see that it's fairly easy to create a Windowed
 * app that can run any REXX script, while routing all messages for errors (not
 * trapped by the script) to a graphical message box. So, if we want to add our own
 * External Functions to let specially written REXX scripts tap into our GUI,
 * we're all set to go, and we still have support for all other REXX scripts. Yeah!
 */

#ifdef WIN32
#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#endif
#include "..\rexxsaa.h"





/* Here is my OPTIONS string passed to RexxSetOptions(). I'm setting the
 * MSGBOX, NULTERM, and CONSOLE options. The order of the items isn't
 * important, but they must be upper case. Also note that the string
 * doesn't need to be nul-terminated.
 */

CHAR MyOptions[] = {'C','O','N','S','O','L','E',' ','N','U','L','T','E','R','M',' ','M','S','G','B','O','X'};






/* Here's our program entry point. It is defined for MS Windows. A C compiler for another
 * operating system will likely require you to declare it with some other name, and with
 * other args passed to it.
 */

int WINAPI WinMain(HINSTANCE  hInstance, HINSTANCE  hPrevInstance, LPSTR lpCmdLine, int nShowCmd)
{
	RXSTRING	retstr;
	APIRET		err;

	/* Set the CONSOLE, MSGBOX, and NULTERM options */
	if (RexxSetOptions(&MyOptions[0], sizeof(MyOptions)))
	{
#ifdef WIN32
		MessageBox(0, "Can't set REXX options", "REXX Error", MB_OK|MB_ICONSTOP);
#else
		/* For other operating systems, here you'd display the error message
		 * "Can't set REXX options".
		 */
#endif
		return(-1);
	}

	/* Pass an RXSTRING to get any possible return value from the script. Let the
	 * interpreter allocate the data buffer
	 */
	retstr.strptr = 0;

	/* Run the REXX Script named "test.rex" in the current directory */
	err = RexxStart(0, 0, "test.rex", 0, 0, RXSUBROUTINE, 0, 0, &retstr);

	/* NOTE: If RexxStart() returned an error, it has already displayed
	 * an error message (in the console window, or in a message box if the
	 * MSGBOX OPTION is set). So we don't need to do that.
	 */

	/* If success, let's check if the script returned anything. We have to
	 * free that if it did.
	 */
	if (!err)
	{
		/* Did the script return some value? */
		if (retstr.strptr && retstr.strlength)
		{
			/* Print out any value returned by the script. NOTE: NULTERM
			 * option nul-terminates the data buffer. Of course, there could
			 * be embedded nul bytes in the data itself. We aren't checking
			 * for that here, since it isn't important to us
			 */
#ifdef WIN32
			MessageBox(0, retstr.strptr, "Script returned:", MB_OK);
#else
			/* For other operating systems, here you'd display the contents of
			 * retstr.strptr in some sort of graphical window you create.
			 * retstr.strptr is nul-terminated because of NULTERM OPTION.
			 */
#endif
			/* Free the data buffer */
			RexxFreeMemory(retstr.strptr);
		}

		/* The script didn't return a value */
		else
		{
#ifdef WIN32
			MessageBox(0, "", "Script did not return a value", MB_OK);
#else
			/* For other operating systems, here you'd display a message that
			 * the script did not return any value.
			 */
#endif
		}
	}

	/* All done! Easy, huh? */
	return(0);
}
