/*
 *  The Reginald Rexx Interpreter
 *  Copyright (C) 1993-1994  Anders Christensen <anders@pvv.unit.no>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Modified by Jeff Glatt (jglatt@borg.com)
 * Nov 5, 2004
 */

#ifndef __REXXSAA_H_INCLUDED
#define __REXXSAA_H_INCLUDED

#define USING_REGINALD

#if defined(WIN32)
/* Stupid MSC can't compile own headers without warning at least in VC 5.0 */
#	if defined(_MSC_VER)
#		if _MSC_VER >= 1100
#			pragma warning(disable: 4115 4201 4214)
#		endif
#		include <windows.h>
#		if _MSC_VER >= 1100
#			pragma warning(default: 4115 4201 4214)
#		endif
#	elif defined(__WATCOMC__) || defined(__BORLANDC__)
#		include <windows.h>
#	endif
#endif
#if defined(__RSXNT__)
#	include <windows.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifndef CONST
#	define CONST const
#endif

#if defined(__MINGW32__)
typedef const char *PCSZ;
#define EXTNAME(n) __attribute__((alias (n)))
#else
#define EXTNAME(n)
#endif

#if !defined(_OS2EMX_H) && !defined(__RSXNT__) && !defined(__MINGW32__)

/* Defines for various datatypes such as CHAR, LONG, PSZ, etc */
typedef char			CHAR;
typedef short			SHORT;
typedef long			LONG;
typedef CHAR *			PSZ;
typedef const CHAR *	PCSZ;
#ifndef VOID
	typedef void		VOID;
#endif
typedef VOID *			PVOID;
typedef CHAR *			PCHAR;
typedef SHORT *			PSHORT;
typedef LONG *			PLONG;
#ifndef UCHAR_TYPEDEFED
 typedef unsigned char	UCHAR;
#endif
#ifndef USHORT_TYPEDEFED
 typedef unsigned short	USHORT;
#endif
#ifndef ULONG_TYPEDEFED
 typedef unsigned long	ULONG;
#endif
typedef USHORT *		PUSHORT;
typedef CHAR *			PCH;
typedef UCHAR *			PUCHAR;

#endif /* _OS2EMX_H && !__RSXNT__ && !__MINGW32__ */

/* Define for APIENTRY, APIRET, and PFN */
#if !defined(APIENTRY)
#	define APIENTRY
#endif

#if !defined(APIRET)
#	define APIRET ULONG
#endif

#if !defined(_OS2EMX_H)
	typedef APIRET (APIENTRY *PFN)();
#endif





/* What follows is the REXX specific portion of the include file,
 * now that the above operating system particular stuff is out of
 * the way.
 */

/* Strings are stored via an RXSTRING struct -- not C-style strings */
typedef struct RXSTRING_type 
{
	ULONG	strlength;
	CHAR	*strptr;
} RXSTRING;
typedef RXSTRING *PRXSTRING;

/* The length of the default string that the interpreter passes to
 * your Exit Handlers and External Functions
 */
#define RXAUTOBUFLEN  256L

/* Macros for RXSTRING manipulation */  
#define MAKERXSTRING(x,c,l)	((x).strptr=(c),(x).strlength=(l))	/* Assign a C-string to an RXSTRING */
#define RXNULLSTRING(x)		(!(x).strptr)						/* Check if an RXSTRING has a "null string" */
#define RXSTRLEN(x)			((x).strptr ? (x).strlength : 0UL)	/* Get the length of an RXSTRING's value string */
#define RXSTRPTR(x)			((x).strptr)						/* Get a pointer to an RXSTRING's value string */
#define RXVALIDSTRING(x)	((x).strptr && (x).strlength)		/* Check that an RXSTRING hasn't a zero-length value string */
#define RXZEROLENSTRING(x)	((x).strptr && !(x).strlength)		/* Check if an RXSTRING has a "zero-length value string" */

/* Structure passed to RexxStart() if you wish to install Exit Handlers */
typedef struct {
	const CHAR	*sysexit_name;
	LONG		sysexit_code;
} RXSYSEXIT;
typedef RXSYSEXIT *PRXSYSEXIT;

/* An interpreter function that frees memory that the interpreter returns to you (ie, the return RXSTRING from RexxStart()) */
APIRET APIENTRY RexxFreeMemory(void *MemoryBlock);
#define REXXFREEMEMORY RexxFreeMemory
EXTNAME("RexxFreeMemory");
typedef APIRET APIENTRY RexxFreeMemoryPtr(PVOID);

/* An interpreter function that allows your app to allocate memory that it returns to
 * the interpreter (ie, from an External Function, Exit, and/or SubCommand Handler).
 * For any memory pointer you return to the interpreter, never let it point to global
 * data in your app (unless Reginald docs specify otherwise), nor allocate your own
 * memory via some other function than RexxAllocateMemory()
 */
void * APIENTRY RexxAllocateMemory(ULONG size);
typedef void * APIENTRY RexxAllocateMemoryPtr(ULONG);

ULONG APIENTRY RexxSetChanges(ULONG changed);
typedef ULONG APIENTRY RexxSetChangesPtr(ULONG);

APIRET APIENTRY RexxRun(void *Ptr, LONG ArgCount, RXSTRING *ArgList, const CHAR *ScriptName);
typedef APIRET APIENTRY RexxRunPtr(void *, LONG, RXSTRING *, const CHAR *);

/* The interpreter function that launches (interprets) a REXX script */
APIRET APIENTRY RexxStart(
	LONG		ArgCount,
	RXSTRING 	*ArgList,
	const CHAR	*ScriptName,
	RXSTRING	*Instore,
	const CHAR	*EnvName,
	LONG		CallType,
	RXSYSEXIT	*Exits,
	SHORT		*ReturnCode,
	RXSTRING	*Result);
typedef APIRET APIENTRY RexxStartPtr(LONG, PRXSTRING, PCSZ, PRXSTRING, PCSZ, LONG, PRXSYSEXIT, PSHORT, PRXSTRING);

/* RexxStart() CallType codes for how to run the script */
#define RXCOMMAND		0	/* Can pass one RXSTRING in ArgList. Script may or may not return a value */
#define RXSUBROUTINE	1	/* Pass any amount of args. Script may or may not return a value */
#define RXFUNCTION		2	/* Pass any amount of args. Script must return a value */

/* 
 * RexxStart() return codes. There doesn't seem to be a standard, except
 * zero is success. The other error codes below are Reginald's. If your program
 * has a console window open, then Reginald displays error messages for all.
 * But for error numbers less than RX_START_SPECIAL, Reginald will not pass these
 * to any RXSIO Exit Handler you specify to RexxStart(). Reginald will display them
 * only to a console window. If your program does not have a console window open,
 * then you will have to display your own error message for one of these errors.
 * (NOTE: These errors are RX_OUT_OF_MEMORY, RX_LOAD_EXITDLL, RX_EXIT_CONFLICT,
 * RX_START_UNIMPL, and RX_DIDNT_START). To help you construct an error message,
 * RexxStart() returns an appropriate, nul-terminated error message in any Result
 * RXSTRING you pass to RexxStart(), as long as you provided the buffer.
 */
#define RX_START_OK			0	/* Success */
#define RX_OUT_OF_MEMORY	1	/* Not enough memory */
#define RX_LOAD_EXITDLL		2	/* Can't auto-load some Exit Handler DLL */
#define RX_EXIT_CONFLICT	3	/* A name conflict with another auto-loaded Exit Handler */
#define RX_START_UNIMPL		4	/* A needed feature is not implemented in this version of REXX */
#define RX_DIDNT_START		5	/* Unable to start REXX */
#define RX_OUT_OF_MEMORY2	11	/* Not enough memory */
#define RX_LOAD_EXITDLL2	12	/* Can't auto-load some Exit Handler DLL */
#define RX_EXIT_CONFLICT2	13	/* A name conflict with another auto-loaded Exit Handler */
#define RX_START_UNIMPL2	14	/* A needed feature is not implemented in this version of REXX */
#define RX_DIDNT_START2		15	/* Unable to start REXX */
#define RX_START_BADP		21	/* Bad parameters */
#define RX_START_TOOMANYP	22	/* Too many args passed to the script. RXCOMMAND allows only 1 */
#define RX_SCRIPT_NOT_FOUND	23	/* Script/Macro can't be located */
#define RX_SCRIPT_ERROR		24	/* Error while running script (including script aborted by RexxSetHalt() or resources failed, or SYNTAX error) */
#define RX_NO_RETURN		25	/* A script invoked as RXFUNCTION did not return a value */
#define RX_NOT_NUMERIC		26	/* Script did not return a whole number */

#define RX_START_SPECIAL	RX_OUT_OF_MEMORY2

/* Suggested limit for Environment name's length. Reginald supports any length (up to 4 million or so chars) */
#define MAXENVNAMELEN 31

/* Reginald-enforced limit for Exit Handler name's length */
#define MAXEXITNAMELEN 254

/* Proprietary Reginald API to test if a word is a REXX function name or keyword.
 * RETURNS: 1 if a keyword, 2 if a function name, 3 if a DO or END keyword,
 * 4 if an IF/THEN/ELSE/WHEN/OTHERWISE keyword, or 0 if none of the preceding.
 */
APIRET APIENTRY RexxIsKeyword(const UCHAR *, ULONG);
typedef APIRET APIENTRY RexxIsKeywordPtr(const UCHAR *, ULONG);

UCHAR * APIENTRY RexxGetInfo(void *, ULONG);
typedef UCHAR * APIENTRY RexxGetInfoPtr(void *, ULONG);

#define RXGETINFO_REXX		0
#define RXGETINFO_NAME		1
#define RXGETINFO_PATH		2
#define RXGETINFO_NUMDIGITS	4





/* ===================== REGINALD PROPRIETARY RAM-BASED SCRIPTS ===================== */

/* Start and End pointers to the start and end of the line in RAM. A line must
 * end with a '\n', '\r', or '\r\n' characters, except for the very last line
 * to be parsed.
 */
typedef struct {
	const UCHAR	*Start;		/* Pointer to start of line */
	const UCHAR	*End;		/* Pointer to end of line (ie, after last UCHAR) */
	ULONG		Breakpoint;	/* RXBREAKPOINT if a breakpoint on this line. Otherwise, RXNOBREAKPOINT. Ignored if you have your own RexxTestBreakPoint callback */
} RAMSOURCE;

#define RXBREAKPOINT	0x80000000
#define RXNOBREAKPOINT	0x00000000

/* The declaration for your "RexxSourceLine callback", a routine you can supply to Reginald
 * via RexxSetSource() to retrieve each source line of a RAM-based script.
 */
typedef LONG APIENTRY RexxSourceLine(void *, RAMSOURCE *, ULONG, const UCHAR *);

/* Returns from RexxSetSource() */
#define RXSOURCE_OK				0
#define RXSOURCE_STILLRUNNING	1	/* Tried to clear your RexxSourceLine callback while a RAM-based script is still running */
#define RXSOURCE_NOTLOADED		2	/* Called RexxGetSource() when no REXX script is loaded/running */
#define RXSOURCE_OUTOFRANGE		3	/* Line number passed to RexxGetSource() is out of range */

APIRET APIENTRY RexxSetSource(RexxSourceLine *);
typedef APIRET APIENTRY RexxSetSourcePtr(RexxSourceLine *);

APIRET APIENTRY RexxGetSource(void *, RAMSOURCE *, ULONG);
typedef APIRET APIENTRY RexxGetSourcePtr(void *, RAMSOURCE *, ULONG);






/* ========================= REGINALD PROPRIETARY OPTIONS ========================= */

/* The proprietary Reginald API to set OPTIONS */
APIRET APIENTRY RexxSetOptions(const CHAR *, ULONG);
typedef APIRET APIENTRY RexxSetOptionsPtr(const CHAR *, ULONG);

/* Returns from RexxSetOptions() */
#define RXOPT_OK		0
#define RXOPT_NOBUILTIN	1	/* Reginald's MSGBOX can't be utilized */






/* ====================== REGINALD PROPRIETARY ERROR DISPLAY ====================== */

/* The structure passed to any "Error Handler" you set via RexxSetErrorHandler() */
typedef struct {
	const UCHAR	*scripttype;	/* Type of REXX script containing the error -- "Script" for disk-based script or "Macro" for instore macro, or 0 if error not related to contents of a REXX script */ 
	const UCHAR	*scriptname;	/* Full name of the REXX script containing the error or 0 if an error not related to the contents of the REXX script */
	const UCHAR	*sourceline;	/* Source line containing the error, or 0 if an error not related to the contents of the REXX script */
	const UCHAR	*helpfile;		/* Filename of online help book, or 0 if none */
	UCHAR		*message;		/* Error message */
	ULONG		lineno;			/* Line # in the source where error occurred or 0 if error not related to contents of the script */
	USHORT		charno;			/* Character # on the line where error occurred, or 0 if error not related to contents of the script */
	UCHAR		GEnum;			/* General Error number. Will be ANSI REXX General Error # if not some host entity calling RexxRaiseCondition */
	UCHAR		SEnum;			/* Sub-Error # */
} REXXERRORMSG;

/* The declaration for your "Error Handler", a routine you can supply to Reginald
 * via RexxSetErrorHandler() to display an untrapped, abortable error condition. It
 * should return RXERR_HANDLED if you display the error message, or RXERR_NOT_HANDLED
 * if you want Reginald to do so.
 */
typedef APIRET APIENTRY RexxErrorHandler(void *rexxPtr, REXXERRORMSG *);

/* The proprietary Reginald API to set your above Error Handler */
APIRET APIENTRY RexxSetErrorHandler(RexxErrorHandler *, RexxErrorHandler **);
typedef APIRET APIENTRY RexxSetErrorHandlerPtr(RexxErrorHandler *, RexxErrorHandler **);

/* Returns from RexxSetErrorHandler() */
#define RXERR_OK		0

/* The proprietary Reginald API to present Reginald's built-in error message box
 * from your RexxErrorHandler
 */
APIRET APIENTRY RexxErrMessageBox(void *, REXXERRORMSG *);
typedef APIRET APIENTRY RexxErrMessageBoxPtr(void *, REXXERRORMSG *);

/* Returns from RexxErrMessageBox() */
#define RXERR_HANDLED		0
#define RXERR_NOT_HANDLED	1

/* Proprietary API to fetch a nul-terminated error message from a REXXSAA API error number */
/*
const UCHAR * APIENTRY RexxGetErrorText(APIRET, const UCHAR *);
typedef const UCHAR * APIENTRY RexxGetErrorTextPtr(APIRET, const UCHAR *);
*/

/* Proprietary API to test the SYNTAX of a REXX Script */
APIRET APIENTRY RexxCheckSyntax(void *, const CHAR *, LONG);
typedef APIRET APIENTRY RexxCheckSyntaxPtr(void *, const CHAR *, LONG);

#define REXXCHECK_ALLERRORS 2
#define REXXCHECK_FIRSTERROR 3






/* ====================== Reginald's Proprietary Debugger API ====================== */

/* Sets/clears a breakpoint. Can also be used to ascertain the line number upon which
 * a breakpoint can be legally set. The first arg is a pointer to the name of the REXX
 * script in which the breakpoint is to be set. This pointer must be the full name
 * pointer passed to your RexxSourceLine() callback by Reginald. The second arg is the
 * desired line number for the breakpoint. RexxSetBreakPoint will return the closest
 * legal line number where that breakpoint may be set. The third arg is one of the
 * flags below.
 */
ULONG APIENTRY RexxSetBreakPoint(ULONG, ULONG);
typedef ULONG APIENTRY RexxSetBreakPointPtr(ULONG, ULONG);

/* Flags for RexxSetBreakPoint() */
#define RXBREAKPOINT_QUERY		1 /* Just check the line number, but do not alter any breakpoint */
#define RXBREAKPOINT_SET		2 /* Set a breakpoint. Applicable only if you don't have a RexxTestBreakPoint callback */
#define RXBREAKPOINT_TOGGLE		4 /* Toggle a breakpoint. Applicable only if you don't have a RexxTestBreakPoint callback */
#define RXBREAKPOINT_CLEAR		8 /* Clear a breakpoint. Applicable only if you don't have a RexxTestBreakPoint callback */
#define RXBREAKPOINT_CLEARALL	0x10 /* Clear all breakpoints. Applicable only if you don't have a RexxTestBreakPoint callback */

/* Struct that REXX passes to your RexxDebugSetup() function */
typedef struct _REXXDEBUGSETUP
{
	UCHAR *ScriptName;  /* Full name of the script currently being debugged. If 0, then debug is off */
	UCHAR CallType;		/* RXCOMMAND, RXSUBROUTINE, or RXFUNCTION */
	UCHAR ScriptType;	/* See below */
	UCHAR Mode;			/* 0 for Run mode, non-zero for Step mode */
} REXXDEBUGSETUP;

/* Struct that REXX passes to your RexxHighlightLine() function */
typedef struct _REXXDEBUGMARK
{
  ULONG LineNumber;	 /* The line number of the line being traced */
  ULONG CharNumber;	 /* The character number of the current step within that line */
  void  *Saved;		 /* The value returned from this level's call to RexxPushDebugLevel() */
  UCHAR Flags;		 /* See below */
} REXXDEBUGMARK;

/* REXXDEBUGMARK Flags */
#define RXDEBUG_UPDATE 0x01		/* Set when returning from a call to another script, macro, or
								 * INTERPRET string. When this is set, you should definitely
								 * recheck that you have the correct source lines loaded/displayed
								 */
#define RXDEBUG_NOCLEAR 0x02	/* Don't clear away the results of the current level. Implement
								 * this if you wish to mimic Reginald's debugger which clears its
								 * RESULTS box of everything printed out since the last call to
								 * RexxPushDebugLevel(). When this flag is set, do not clear those
								 * results
								 */

/* Struct that REXX passes to your RexxPushDebugLevel() function */
typedef struct _REXXDEBUGINLEVEL
{
	const CHAR *ScriptName; /* If another REXX script is being called, this is the full name
						* of that script. If 0, then only a subroutine inside the current
						* script is being called, or a built-in/external function called.
						*/
	const CHAR *FuncName; /* The nul-terminated name of the subroutine/script as it is notated
					  * in the script. This is what Reginald's debugger displays in its RESULTS
					  * box as the name of the subroutine/function/script being called.
					  */
	UCHAR	CallType;	/* RXCOMMAND, RXSUBROUTINE, or RXFUNCTION */
	UCHAR	ScriptType; /* See below */
	UCHAR	StepOver;	/* 1 if stepping over the function */
} REXXDEBUGINLEVEL;


/* Struct that REXX passes to your RexxPopDebugLevel() function */
typedef struct _REXXDEBUGOUTLEVEL
{
	const CHAR *FromScriptName; /* The full name of the script that the returning subroutine
						    * is in. (ie, The script that has just finished).
							*/
	const CHAR *ToScriptName;   /* The full name of the script that the contains the call to
							* the subroutine. (ie, The script that is now being returned
							* to -- is being resumed).
							*/
	void  *Saved;		/* The value returned from the respective call to RexxPushDebugLevel() */
	UCHAR CallType;		/* RXCOMMAND, RXSUBROUTINE, or RXFUNCTION */
	UCHAR ScriptType;	/* See below */
} REXXDEBUGOUTLEVEL;

/* For ScriptType */
#define DEBUGTYPE_MACRO		0x01	/* An instore REXX macro */
#define DEBUGTYPE_SCRIPT	0x02	/* An external REXX script */
#define DEBUGTYPE_NONREXX	0x04	/* A built-in or external DLL/EXE (non-REXX) function */
#define DEBUGTYPE_INTERNAL	0x08	/* An internal REXX subroutine/function within the script */
#define DEBUGTYPE_STRING	0x10	/* An INTERPRET string */
#define DEBUGTYPE_SHIFT		3

/* To replace Reginald's built-in debugger with your own, use RexxInstallDebug()
 * to install your debugger callbacks. Put pointers to your callbacks in the
 * REXXDEBUGPTRS struct. You must also install an RXSIO Exit Handler and handle
 * RXSIODTR if you want to do your own GetMessage()/DispatchMessage() loop for
 * your debugger window.
 */
typedef struct _REXXDEBUGPTRS
{
	void * (APIENTRY *RexxDebugSetup)(void *, REXXDEBUGSETUP *); /* Called by REXX whenever debugging is turned on/off.
												 * Should return a handle to your debugger window, or
												 * 0 if an error
												 */
	void * (APIENTRY *RexxPushDebugLevel)(void *, REXXDEBUGINLEVEL *); /* Called by REXX whenever a new level is entered.
												 * Should return some value that is stored by REXX
												 * and passed to your RexxDebugPopLevel(), or 0 if
												 * an error.
												 */
	BOOL (APIENTRY *RexxPopDebugLevel)(void *, REXXDEBUGOUTLEVEL *); /* Called by REXX whenever a new level is exited.
												 * Should return TRUE if successful or FALSE if an
												 * error.
												 */
	BOOL (APIENTRY *RexxPrintDebugResult)(const UCHAR *);	/* Called by REXX whenever a nul-terminated message
												 * must be printed to your debugger's RESULTS area.
												 * Should return TRUE if successful or FALSE if an
												 * error.
												 */
	APIRET (APIENTRY *RexxHighlightLine)(REXXDEBUGMARK *);	/* Called by REXX whenever the currently executing
												* source line needs to be highlighted in the
												* debugger window. This should return one of the
												* values listed below.
												*/
	APIRET (APIENTRY *RexxMessagePump)(void);	/* Called by REXX whenever the debugger window should
											 * service its message queue. This should check for the
											 * user wanting to break a "run" or quit, and should
											 * return one of the values listed below.
											 */
	APIRET (APIENTRY *RexxGetDebugInput)(void *); /* Called by REXX to get user input whenever
												   * the debugger has paused the script at a
												   * breakpoint. Should return one of the
												   * RXDEBUG_ error numbers listed below.
												   */
	UCHAR (APIENTRY *RexxTestBreakPoint)(const UCHAR *, ULONG);	/* Called by REXX at the start of interpreting
												* each line to see if there is a breakpoint set. It
												* should return TRUE to break or FALSE if no break.
												*/

	APIRET (APIENTRY *RexxWatch)(void *, const UCHAR *);	/* Called by REXX whenever a variable's
												* value changes. It should return TRUE to break or FALSE if no break.
												*/
} REXXDEBUGPTRS;

/* This is the struct actually passed to RexxInstallDebug(). You must set the
 * size field of it to sizeof(REXXDEBUGPTRS). Then you must fill in its
 * REXXDEBUGPTRS struct.
 */

typedef struct _REXXDEBUGGER
{
	ULONG					size;
	struct _REXXDEBUGPTRS	ptrs;
} REXXDEBUGGER;

APIRET APIENTRY RexxInstallDebug(REXXDEBUGGER *);
typedef APIRET APIENTRY RexxInstallDebugPtr(REXXDEBUGGER *);

/* Return codes from RexxInstallDebug() */
#define RXDEBUG_OK		0		/* Success */

/* Return codes from your RexxMessagePump() and RexxHighlightLine() functions */
#define RXDEBUG_HANDLED	0		/* You have successfully handled the operation, and wish Reginald
								 * to proceed with normal debugging, and no default action.
								 */
#define RXDEBUG_BREAK	1		/* Force a break at end of the currently running instruction
								 * (whether or not there is a breakpoint set there) and
								 * call your RexxGetDebugInput() function.
								 */
#define RXDEBUG_NOT_HANDLED	2	/* For RexxHighlightLine(): Do the default debugger action of
								 * causing your debugger window to redraw itself and clear your
								 * RESULTS box. For RexxMessagePump(): Service any queued messages
								 * in the debugger queue.
								 */
/* RexxMessagePump() and RexxHighlightLine() can also return REXDEBUG_OUTOFMEM and RXDEBUG_ABORT */
 
/* Return codes from your RexxGetDebugInput() function */
#define RXDEBUG_OUTOFMEM	-1		/* A memory error */
#define RXDEBUG_ABORT		-2		/* Abort the script */
#define RXDEBUG_RUN			1		/* Initiate "Run" mode (ie, run until the next breakpoint) */
#define RXDEBUG_STEP		3		/* Initiate "Step" mode (ie, execute the next step and then break) */
#define RXDEBUG_STEPOVER	6		/* Initiate "Step Over" mode (ie, execute the next step and then break, but don't dive into any sub-routine, script, INTERPRET string) */
#define RXDEBUG_REDO		0x13	/* Redo current instruction and then break */
#define RXDEBUG_INPUT		0x23	/* Interpret returned string and then break */
#define RXDEBUG_CLOSE		0x81	/* Close the debugger, and run the script as normally */
#define RXDEBUG_SETTINGS	0x82	/* Settings were changed on-the-fly */

APIRET APIENTRY RexxDebuggerBreak(ULONG halt); /* halt = 1 to abort script, 0 to break */
typedef APIRET APIENTRY RexxDebuggerBreakPtr(ULONG);

APIRET APIENTRY RexxSetTrace(void *, CHAR *, ULONG, ULONG);
typedef UCHAR * APIENTRY RexxSetTracePtr(void *, CHAR *, ULONG, ULONG);






/* ====================== SYSTEM EXIT HANDLERS ====================== */

/*
 * The Function Codes (and their associated SubCodes) passed to your Exit
 * Handler. All RXxxx symbols are Function Codes, while the 
 * RXxxxyyy symbols are Function SubCodes for Function Code RXxxx.
 */
#define RXFNC		2	/* Function Exit Handler -- Called before every External Function is called */
# define RXFNCCAL	1

#define RXCMD		3	/* Subcom Exit Handler -- Called before each command is passed to an external Environment (Subcom Handler) */
# define RXCMDHST	1

#define RXMSQ		4	/* Queue Exit Handler -- Called when a line needs to be PUSH'ed or PULL'ed from a queue */
# define RXMSQPLL	1	/* Called when a line is PULL'ed */
# define RXMSQPSH	2	/* Called when a line is PUSH'ed or QUEUE'ed */
# define RXMSQSIZ	3	/* Called when the # of lines in a queue are queried, ie, QUEUED() */
# define RXMSQNAM	20	/* Called when the name of the queue is set with RXQUEUE('S') */

#define RXSIO		5	/* String I/O Exit Handler -- Called whenever a line is output for tracing or SAY, or a line needs to be gotten from the user */
# define RXSIOSAY	1	/* Called when a line is SAY'ed */
# define RXSIOTRC	2	/* Called when a line is traced or an error message displayed */
# define RXSIOTRD	3	/* Called when a line is gotten from the user as a result of a PULL on an empty queue */
# define RXSIODTR	4	/* Called when a line is gotten from the user for interactive tracing */
# define RXSIOTLL	5	/* Not yet implemented */
# define RXSIOTRDN	6	/* Called when a requested number of characters need to be gotten from the user */
# define RXSIOSAYN	7	/* Called when a requested number of characters need to be displayed */

#define RXHLT		7	/* Halt Exit Handler -- Called whenever a Halt condition is about to be raised?? */
# define RXHLTCLR	1	/* Called when a HALT condition is being triggered (but not yet handled by the script) */
# define RXHLTTST	2

#define RXTRC		8	/* ??? */
# define RXTRCTST	1

#define RXINI		9	/* Init Exit Handler -- Called right before a script is interpreted */
# define RXINIEXT	1

#define RXTER		10	/* Termination Exit Handler -- Called right after a script is interpreted */
# define RXTEREXT	1

#define RXDBG		11	/* Debug Exit Handler??? */
# define RXDBGTST	1

#define RXENV		12	/* System Environment variable handling */
# define RXENVGET	1	/* Get System Environment Variable */
# define RXENVSET	2	/* Set System Environment Variable */

#define RXENDLST	0	/* Marks the last RXSYSEXIT in the array passed to RexxStart() */
#define RXNOOFEXITS	13	/* Total # of Exit Handler types (counting unused values) */
  
/* Structure that the interpreter passes to your Exit Handler
 * which contains information from the interpreter, or in which
 * you return information to the interpreter
 */
typedef struct {
   struct {
		unsigned rxfferr:1;
		unsigned rxffnfnd:1;
		unsigned rxffsub:1;
   } rxfnc_flags;
	const CHAR	*rxfnc_name;
	USHORT		rxfnc_namel;
	const CHAR	*rxfnc_que;
	USHORT		rxfnc_quel;
	USHORT		rxfnc_argc;
	RXSTRING	*rxfnc_argv;	/* const */ 
	RXSTRING	rxfnc_retc;
} RXFNCCAL_PARM;			/* Used for RXFNC type */

typedef struct {
	struct {
		unsigned rxfcfail:1;
		unsigned rxfcerr:1;
	} rxcmd_flags;
	const CHAR	*rxcmd_address;
	USHORT		rxcmd_addressl;
	const CHAR	*rxcmd_dll;
	USHORT		rxcmd_dll_len;
	RXSTRING	rxcmd_command;	/* const */ 
	RXSTRING	rxcmd_retc;
} RXCMDHST_PARM;			/* Used for RXCMD type */

typedef struct {
	RXSTRING rxmsq_retc;
} RXMSQPLL_PARM;			/* Used for RXMSQ type's RXMSQPLL */

typedef struct {
	struct {
		unsigned rcfmlifo:1;
	} rcmsq_flags;
	RXSTRING rxmsq_value;	/* const */ 
} RXMSQPSH_PARM;			/* Used for RXMSQ type's RXMSQPSH */

typedef struct {
	ULONG rxmsq_size;
} RXMSQSIZ_PARM;			/* Used for RXMSQ type's RXMSQSIZ */

typedef struct {
	RXSTRING rxmsq_name;
} RXMSQNAM_PARM;			/* Used for RXMSQ type's RXMSQNAM */

typedef struct {
	RXSTRING rxsio_string;	/* const */ 
} RXSIOSAY_PARM;			/* Used for RXSIO type's RXSIOSAY */

typedef struct {
	RXSTRING rxsio_string;	/* const */ 
} RXSIOTRC_PARM;			/* Used for RXSIO type's RXSIOTRC */

typedef struct {
	RXSTRING rxsiotrd_retc;
} RXSIOTRD_PARM;			/* Used for RXSIO type's RXSIOTRD or RXSIOTRDN. For the latter, rxsiotrd_retc.strlength is the exact number of bytes that must be returned */

typedef struct {
	RXSTRING rxsiodtr_retc;
} RXSIODTR_PARM;			/* Used for RXSIO type's RXSIODTR */

typedef struct {
	struct {
		unsigned rxfhhalt:1;
	} rxhlt_flags;
} RXHLTTST_PARM;			/* Used for RXHLT type's RXHLTTST */

typedef struct {
	struct {
		unsigned rxftrace:1;
	} rxtrx_flags;
} RXTRCTST_PARM;			/* Used for RXTRC type's RXTRCTST */

typedef struct {
	RXSTRING rxenv_name;
	RXSTRING rxenv_value;
} RXENVGET_PARM;			/* Used for RXENV type's RXENVGET */

typedef struct {
	RXSTRING rxenv_name;
	RXSTRING rxenv_value;
} RXENVSET_PARM;			/* Used for RXENV type's RXENVSET */

typedef union {
	RXFNCCAL_PARM fnccal;
	RXCMDHST_PARM cmdhst;
	RXMSQPLL_PARM msqpll;
	RXMSQPSH_PARM msqpsh;
	RXMSQSIZ_PARM msqsiz;
	RXMSQNAM_PARM msqnam;
	RXSIOSAY_PARM siosay;
	RXSIOTRC_PARM siotrc;
	RXSIOTRD_PARM siotrd;
	RXSIODTR_PARM siodtr;
	RXHLTTST_PARM hlttst;
	RXTRCTST_PARM trctst;
	RXENVGET_PARM envget;
	RXENVSET_PARM envset;
} EXIT;

#ifdef UNION_EXIT
	typedef EXIT *	PEXIT;
#else
	typedef UCHAR	*PEXIT;
#endif

/* The declaration for any Exit Handlers you write. The interpreter calls
 * your Exit Handler with 3 arguments. The first arg is the Function Code.
 * It tells what type of Exit Handler the interpreter is calling. The
 * second arg is the SubCode. It tells what kind of action it wishes your
 * Exit Handler to take. The third arg is a pointer to a structure that
 * contains additional information passed to your Exit Handler, or where
 * you are supposed to return additional information to the interpreter
 * (depending upon what action you're supposed to take). This structure
 * will be different for each type of Exit Handler
 */
typedef APIRET APIENTRY RexxExitHandler(LONG, LONG, /* const */ PEXIT);
typedef RexxExitHandler * RexxRegisterHandlersPtr;

/* Return codes for Exit Handlers (ie, what your Exit Handler returns) */
/* First two are also returns from RexxErrMessageBox() */
#define RXEXIT_HANDLED		0		/* Return this if you handle things yourself and don't want the interpreter to invoke its default action */
#define RXEXIT_NOT_HANDLED	1		/* Return this if you want the interpreter to invoke its default action */
#define RXEXIT_RAISE_ERROR	(-1)	/* Return this if you want the interpreter to raise the SYNTAX condition with error #48 and invoke no other action */

/* Reginald supports auto-loading Exit Handlers in DLLs. To enable an Exit
 * Handler DLL to auto-load, you must include a callable function in it
 * named RexxRegisterHandlers. This is the declaration of it.
 */
APIRET APIENTRY RexxRegisterHandlers(RexxExitHandler *exits[], ULONG numofexits);

/* Return Codes from the interpreter's Exit functions (such as RexxRegisterExitExe() */
#define RXEXIT_ERROR		0x01	/* Not used */
#define RXEXIT_FAILURE		0x02	/* Not used */
#define RXEXIT_ISREG		0x01	/* Function is already registered */
#define RXEXIT_BADENTRY		1001	/* Not used */
#define RXEXIT_NOEMEM		1002	/* Out of memory */
#define RXEXIT_BADTYPE		1003	/* Bad arguments */
#define RXEXIT_NOTINIT		1004	/* External functions not supported -- Only used in non-DLL version */
#define RXEXIT_OK				0
#define RXEXIT_DUP				10	/* Not used */
#define RXEXIT_MAXREG			20	/* Not used */
#define RXEXIT_NOTREG			30	/* Function is not yet registered */
#define RXEXIT_NOCANDROP		40	/* Not used */
#define RXEXIT_LOADERR			50	/* Error loading DLL */
#define RXEXIT_NOPROC			127	/* Function not found in DLL */

/* To register an Exit Handler in an EXE */
APIRET APIENTRY RexxRegisterExitExe(const CHAR *EnvName,
#ifdef RX_STRONGTYPING
   RexxExitHandler *EntryPoint,
#else
   PFN EntryPoint, 
#endif
   PUCHAR UserArea);
#define REXXREGISTEREXITEXE RexxRegisterExitExe
EXTNAME("RexxRegisterExitExe");
typedef APIRET APIENTRY RexxRegisterExitExePtr(const CHAR *,
#ifdef RX_STRONGTYPING
   RexxExitHandler *,
#else
   PFN,
#endif
PUCHAR);

/* To register an Exit Handler in a DLL */
APIRET APIENTRY RexxRegisterExitDll(const CHAR *EnvName, const CHAR *ModuleName, const CHAR *ProcedureName, UCHAR *UserArea, ULONG DropAuth);
typedef APIRET APIENTRY RexxRegisterExitDllPtr(const CHAR *, const CHAR *, const CHAR *, UCHAR *, ULONG);

/* Value of DropAuth passed to RexxRegisterExitDll() */
#define RXEXIT_DROPPABLE	0x00	/* The interpreter may deregister the function on its own after running the REXX script */
#define RXEXIT_NONDROP		0x01	/* The interpreter may not deregister the function on its own. Your DLL must specifically call RexxDeregisterExit() */

/* To Deregister (ie, remove) an Exit Handler in a DLL or EXE */
APIRET APIENTRY RexxDeregisterExit(const CHAR *EnvName, const CHAR *ModuleName);
typedef APIRET APIENTRY RexxDeregisterExitPtr(const CHAR *, const CHAR *);

/* To verify if an Exit Handler in a DLL or EXE is already registered */
APIRET APIENTRY RexxQueryExit(const CHAR *ExitName, const CHAR *ModuleName, USHORT *Flag, UCHAR *UserArea);
typedef APIRET APIENTRY RexxQueryExitPtr(const CHAR *, const CHAR *, USHORT *, UCHAR *);

typedef APIRET APIENTRY RexxSetReginaldExitsPtr(void *);

/* The declaration for your "SAY Handler", a routine you can supply to Reginald
 * via RexxSetErrorHandler() to display a string output via SAY
 */
typedef void APIENTRY RexxSayHandler(void *, const UCHAR *, ULONG, ULONG);

/* The declaration for your "PULL Handler", a routine you can supply to Reginald
 * via RexxSetErrorHandler() to return a string input via PULL. The string you
 * return should be allocated via RexxGetBuffer()
 */
typedef const UCHAR * APIENTRY RexxPullHandler(void *, ULONG);

/* The declaration for your "ENVIRONMENT Handler", a routine you can supply to Reginald
 * via RexxSetErrorHandler() to return the value of some environment variable
 */
typedef APIRET APIENTRY RexxEnvHandler(const UCHAR *, UCHAR **);

typedef struct {
	RexxPullHandler *HostPullHandler;
	RexxSayHandler *HostSayHandler;
	RexxEnvHandler *HostEnvHandler;
} EXITHANDLERS;







/* ======================== VARIABLE POOL ======================== */

/*
 * NOTE: To retrieve the version of Reginald, use the RexxVariablePool()
 * to fetch the special variable "VERSION". (ie, Pass a SHVBLOCK with
 * its 'shvcode' field set to RXSHV_PRIV and its 'shvname' RXSTRING
 * set to "VERSION"). This will return a string beginning with the
 * letters "REXX-", immediately followed by the name of the interpreter
 * (which is "Reginald"), immediately followed by an underscore and
 * then the version/revision numbers separated by a period. Then, there
 * will be a space followed by the REXX language level supported by the
 * interpreter. There will follow another space, and then the date that
 * the interpreter was released. For example:
 *
 *			REXX-Reginald_1.0 4.00 12 Jul 2001
 */

/* Return codes from RexxVariablePool() */
# define RXSHV_OK		0x00	/* Everything OK */
# define RXSHV_NEWV		0x01	/* Variable did not previously exist before this operation */
# define RXSHV_LVAR		0x02	/* Last variable in a NEXTV sequence */
# define RXSHV_TRUNC	0x04	/* Name or value has been truncated */
# define RXSHV_BADN		0x08	/* Bad/invalid variable name passed to RexxVariablePool() */
# define RXSHV_MEMFL	0x10	/* Memory problem, e.g. out of memory */
# define RXSHV_BADF		0x80	/* Invalid function code passed to RexxVariablePool() */

# define RXSHV_NOAVL	0x90	/* The interpreter returns this for operations it does not yet support */


/* Function (operation) codes passed to RexxVariablePool() */
# define RXSHV_SET		0x00	/* Set variable's value */
# define RXSHV_FETCH	0x01	/* Get value of variable */
# define RXSHV_DROPV	0x02	/* Drop variable */
# define RXSHV_SYSET    0x03	/* Set symbolic variable's value */
# define RXSHV_SYFET    0x04	/* Get value of symbolic variable */
# define RXSHV_SYDRO    0x05	/* Drop symbolic variable */
# define RXSHV_NEXTV	0x06	/* Query next variable in a NEXTV sequence */
# define RXSHV_PRIV		0x07	/* Query private information */
# define RXSHV_EXIT		0x08	/* Set function exit value */

/*
 * Note: A symbolic variable is a symbol that will be 'expanded' before 
 *       use, while a (normal) variable is used directly, without 
 *       expanding it first. 
 */

/* Structure passed to RexxVariablePool() which contains information you
 * send to the interpreter in order to perform a set/query/drop operation,
 * or which the interpreter uses to return data to you in order to perform
 * a get operation.
 */
typedef struct shvnode 
{
	struct shvnode	*shvnext;
	RXSTRING		shvname;
	RXSTRING		shvvalue;
	ULONG			shvnamelen;
	ULONG			shvvaluelen;
	UCHAR			shvcode;
	UCHAR			shvret;
} SHVBLOCK;
typedef SHVBLOCK *PSHVBLOCK;

/* The interpreter function to set/query variables */
APIRET APIENTRY RexxVariablePool(PSHVBLOCK);
typedef APIRET APIENTRY RexxVariablePoolPtr(PSHVBLOCK);

/* Reginald Proprietary function to allocate a buffer
 * that can be passed to RexxSetValue(). Passed the
 * desired size of the buffer, and returns a pointer to
 * it, or 0 if success. You may then fill in the buffer
 * with the desired contents and pass it to RexxSetValue.
 * A buffer passed to RexxSetValue must not be freed, nor
 * subsequently referenced/altered
 */
void * APIENTRY RexxGetBuffer(void *, const UCHAR *ptr, ULONG size, ULONG flags);
typedef PVOID APIENTRY RexxGetBufferPtr(void *, const UCHAR *, ULONG, ULONG);

#define RXGET_GRABIT		0x01000000
#define RXGET_TRIMLEADING	0x02000000
#define RXGET_UPPER			0x04000000
#define RXGET_OBJNAME		0x08000000
#define RXGET_STR			0x10000000
#define RXGET_ERR			0x20000000
#define RXGET_INT			0x40000000
#define RXGET_UINT			0x80000000

void APIENTRY RexxFreeBuffer(void *);
typedef void APIENTRY RexxFreeBufferPtr(void *);

/* Reginald Proprietary function to set and fetch a variable's value
 * 'value' must be allocated via RexxGetBuffer
 */
APIRET APIENTRY RexxSetValue(void *, const void *name, const void *objectHandle, const void * value, ULONG flags);
typedef APIRET APIENTRY RexxSetValuePtr(void *, const void *, const void *, const void *, ULONG);

void * APIENTRY RexxGetValue(void *, const void *name, const void *objectHandle, ULONG *size, ULONG flags);
typedef void * APIENTRY RexxGetValuePtr(void *, const void *, const void *, ULONG *, ULONG);

#define RXVAL_DIR		0x00200000
#define RXVAL_NODOT		0x00400000
#define RXVAL_EOLTERM	0x00800000
#define RXVAL_NULTERM	0x01000000
#define RXVAL_INT		0x02000000
#define RXVAL_UINT		0x04000000
#define RXVAL_COMPOUND	0x08000000
#define RXVAL_PARENT	0x10000000
#define RXVAL_NAMECNT	RXNAME_CNT /* name is not nul-terminated, and length is in low word of this same flags arg. */
#define RXVAL_NAMENUL	RXNAME_NUL /* name is nul-terminated */
#define RXVAL_NAMELEN	RXNAME_LEN /* name is not nul-terminated, and begins with a length byte. */

UCHAR * APIENTRY RexxLegalVarName(UCHAR *name, ULONG flags);
typedef UCHAR * APIENTRY RexxLegalVarNamePtr(UCHAR *, ULONG);

#define RXLEGAL_GOTDOT		0x10000000	/* Do not set this */
#define RXLEGAL_TRIMDOTS	0x20000000
#define RXLEGAL_NOPREFIX	0x40000000
#define RXLEGAL_START		0x80000000






/* ====================== SUBCOMMAND HANDLERS ====================== */

/* The declaration for any Subcom Handler you write. The interpreter calls
   your Subcom Handler with 3 arguments. The first arg is a pointer to the
   nul-terminated command string that you should operate upon. The second
   arg is a pointer to where you return your error code which is one of
   the values below or RXSUBCOM_OK if success. The third arg is a pointer
   to an RXSTRING structure that provides a default buffer into which you
   can return some string to the script. Simply set the RXSTRING's strlength
   field to the number of characters you're returning in that default buffer,
   or 0 if you're not returning any string. That string will be assigned to
   the special REXX variable named "RC" (ie, so the script can retrieve it).
   If you need a larger buffer than what is provided (ie, RXAUTOBUFLEN),
   then allocate a buffer with RexxAllocateMemory() and stuff the pointer
   into the provided RXSTRING's strptr field.
 */
typedef APIRET APIENTRY RexxSubcomHandler(/* const */ RXSTRING *, USHORT *, RXSTRING *);

/* Return codes for Subcom Handlers (ie, what your Subcom Handler returns) */
/* I don't know the 'real' values of these */
/* #define RXSUBCOM_OK		0 */	/* For success. You may, may not, choose to use the provided RXSTRING to return some string to the interpreter */
#define RXSUBCOM_ERROR		0x01	/* For failure, and raising the ERROR condition. You should use the provided RXSTRING to return an error message */
#define RXSUBCOM_FAILURE	0x02	/* For failure, and raising the FAILURE condition. You should use the provided RXSTRING to return an error message */

/* To register a Subcom Handler in a EXE */
APIRET APIENTRY RexxRegisterSubcomExe(const CHAR *EnvName,
#ifdef RX_STRONGTYPING
	RexxSubcomHandler *EntryPoint,
#else
	PFN EntryPoint,
#endif
	UCHAR *UserArea);
typedef APIRET APIENTRY RexxRegisterSubcomExePtr(const CHAR *,
#ifdef RX_STRONGTYPING
	RexxSubcomHandler *,
#else
	PFN,
#endif
	UCHAR *);

/* To register a Subcom Handler in a DLL */
APIRET APIENTRY RexxRegisterSubcomDll(const CHAR *EnvName, const CHAR *ModuleName, const CHAR *ProcedureName, UCHAR *UserArea, ULONG DropAuth);
typedef APIRET APIENTRY RexxRegisterSubcomDllPtr(const CHAR *, const CHAR *, const CHAR *, UCHAR *, ULONG);

/* Value of DropAuth passed to RexxRegisterSubcomDll() */
#define RXSUBCOM_DROPPABLE	0x00	/* The interpreter may deregister the function on its own after running the REXX script */
#define RXSUBCOM_NONDROP	0x01	/* The interpreter may not deregister the function on its own. Your DLL must specifically call RexxDeregisterSubcom() */

/* To Deregister (remove) a Subcom Handler in a DLL or EXE */
APIRET APIENTRY RexxDeregisterSubcom(const CHAR *EnvName, const CHAR *ModuleName);
typedef APIRET APIENTRY RexxDeregisterSubcomPtr(const CHAR *, const CHAR *);

/* To verify if a Subcom Handler in a DLL or EXE is already registered */
/* NOTE: The Flag arg of this is ambiguous. Different interpreters may use it differently. Just set it to 0. */
APIRET APIENTRY RexxQuerySubcom(const CHAR *Envname, const CHAR *ModuleName, USHORT *Flag, UCHAR *UserArea);
typedef APIRET APIENTRY RexxQuerySubcomPtr(const CHAR *, const CHAR *, USHORT *, UCHAR *);

/* Return Codes from the interpreter's SubCom API (such as RexxRegisterSubcomExe() */
#define RXSUBCOM_OK			0
#define RXSUBCOM_ISREG		0x01
#define RXSUBCOM_DUP		10
#define RXSUBCOM_MAXREG		20
#define RXSUBCOM_NOTREG		30
#define RXSUBCOM_NOCANDROP	40
#define RXSUBCOM_LOADERR	50
#define RXSUBCOM_NOPROC		127
#define RXSUBCOM_BADENTRY	1001
#define RXSUBCOM_NOEMEM		1002
#define RXSUBCOM_BADTYPE	1003
#define RXSUBCOM_NOTINIT	1004










/* =================== EXTERNAL FUNCTION HANDLERS =================== */

/* The declaration for any External Function Handlers you write. It should return
 * either RXFUNC_OK for success, or one of the General Error numbers below.
 */
typedef APIRET APIENTRY RexxFunctionHandler(const CHAR *, ULONG, /* const */ RXSTRING *, const UCHAR *, RXSTRING *);

/* ANSI General Error (GE) numbers that your function can return */
#define RXERR_INIT_FAIL			3	/* Initialization failed */
#define RXERR_PROG_INTERRUPT	4	/* Execution was aborted by some entity */
#define RXERR_STORAGE_EXHAUSTED	5	/* Out of memory */
#define RXERR_UNMATCHED_QUOTE	6	/* Unmatched quote in an arg */
#define RXERR_FILE				11	/* File system error */
#define RXERR_INVALID_CHAR		13	/* Invalid character in an arg */
#define RXERR_INVALID_HEX_CONST	15	/* Invalid hexadecimal or binary number in arg */
#define RXERR_STRING_EXPECTED	19	/* String or symbol expected in arg */
#define RXERR_SYMBOL_EXPECTED	20	/* (Variable) Name expected in arg */
#define RXERR_INVALID_STR		22	/* Arg is an invalid character string */
#define RXERR_INVALID_DATA_STR	23	/* Arg is an invalid data string */
#define RXERR_INV_SUBKEYWORD	25	/* Invalid sub-keyword found */
#define RXERR_INVALID_INTEGER	26	/* Arg is not a whole number */
#define RXERR_TOO_LONG_STRING	30	/* Name or String too long */
#define RXERR_INVALID_START		31	/* Name starts with number or "." */
#define RXERR_INVALID_RESULT	33	/* Invalid expression result */
#define RXERR_UNLOGICAL_VALUE	34	/* Logical value not 0 or 1 */
#define RXERR_INVALID_EXPRESSION 35	/* Invalid expression */
#define RXERR_INVALID_TEMPLATE	38	/* Invalid template or pattern */
#define RXERR_INCORRECT_CALL	40	/* Incorrect call to routine */
#define RXERR_BAD_ARITHMETIC	41	/* Bad arithmetic conversion */
#define RXERR_ARITH_OVERFLOW	42	/* Arithmetic Overflow/Underflow */
#define RXERR_ROUTINE_NOT_FOUND	43	/* Routine not found */
#define RXERR_NO_DATA_RETURNED	44	/* Function did not return data */
#define RXERR_DATA_NOT_SPEC		45	/* REXX script did not return a value */
#define RXERR_INVALID_VAR		46	/* Invalid variable reference */
#define RXERR_SYSTEM_FAIL		48	/* Operating system failure */
#define RXERR_INTERPRETER_FAIL	49	/* REXX interpreter failure */
#define RXERR_INVALID_FUNC		51	/* Invalid function name */
#define RXERR_INVALID_STEM		54	/* Invalid stem variable */

#define RXERR_EXTFUNC_ERR		0x00010028	/* Return your error message in the provided RXSTRING passed to your function */

/* For these error numbers, you should OR them with MAKEFUNCARG(arg)
 * where "arg" is a decimal value as follows:
 */
#define RXERR_TOOFEWARGS		0x00030028	/* Too few args. "arg" is the min # of args your function accepts */
#define RXERR_TOOMANYARGS		0x00040028	/* Too many args. "arg" is the max # of args your function accepts */
#define RXERR_ARGMISSING		0x00050028	/* A required arg is missing (ie, omitted). "arg" is the arg number missing (where 1 is the first arg) */
#define RXERR_ARGNOTNUL			0x00150028	/* A required arg is an empty string. "arg" is the arg number omitted (where 1 is the first arg) */
#define RXERR_ARGNOTNUM			0x000B0028	/* Arg must have a numeric value. "arg" is the arg number specified incorrectly (where 1 is the first arg) */
#define RXERR_ARGWHOLENUM		0x000C0028	/* Arg must be a whole number (ie, no fractional part). "arg" is the arg number specified incorrectly (where 1 is the first arg) */
#define RXERR_ARGZEROPOS		0x000D0028	/* Arg must be 0 or positive (ie, not negative). "arg" is the arg number specified incorrectly (where 1 is the first arg) */
#define RXERR_ARGPOS			0x000E0028	/* Arg must be positive (ie, not 0 or negative). "arg" is the arg number specified incorrectly (where 1 is the first arg) */
#define RXERR_ARGZEROONE		0x00270028	/* Arg must be 0 or 1. "arg" is the arg number specified incorrectly (where 1 is the first arg) */
#define RXERR_ONECHAR			0x00170028	/* Arg must be a single character. "arg" is the arg number specified incorrectly (where 1 is the first arg) */
#define RXERR_ARGBIN			0x00180028	/* Arg must be a binary number. "arg" is the arg number specified incorrectly (where 1 is the first arg) */
#define RXERR_ARGHEX			0x00190028	/* Arg must be a hex number. "arg" is the arg number specified incorrectly (where 1 is the first arg) */
#define RXERR_SYM				0x001A0028	/* Arg must be a valid symbol. "arg" is the arg number specified incorrectly (where 1 is the first arg) */
#define RXERR_FILENAME			0x001B0028	/* Arg must be a valid or existing filename. "arg" is the arg number specified incorrectly (where 1 is the first arg) */
#define RXERR_VARNAME			0x00240028	/* Arg must be an existing variable. "arg" is the arg number specified incorrectly (where 1 is the first arg) */
#define RXERR_POOLNAME			0x00250028	/* Arg must be the name of a variable pool. "arg" is the arg number specified incorrectly (where 1 is the first arg) */
#define RXERR_INVOPT			0x001C0028	/* Unknown option string. "arg" is the arg number specified incorrectly (where 1 is the first arg). A string listing all accepted options should be copied to the return string, which each option separated by a space */

#define MAKEFUNCARG(x)			(x << 8)

/* To register an External Function in an EXE */
APIRET APIENTRY RexxRegisterFunctionExe(const CHAR *name,
#ifdef RX_STRONGTYPING
   RexxFunctionHandler *EntryPoint);
#else
   PFN EntryPoint);
#endif
typedef APIRET APIENTRY RexxRegisterFunctionExePtr(const CHAR *,
#ifdef RX_STRONGTYPING
   RexxFunctionHandler *);
#else
   PFN);
#endif

/* To register an External Function in a DLL */
APIRET APIENTRY RexxRegisterFunctionDll(const CHAR *ExternalName, const CHAR *LibraryName, const CHAR *InternalName);
typedef APIRET APIENTRY RexxRegisterFunctionDllPtr(const CHAR *, const CHAR *, const CHAR *);

/* To deregister an External Function in an EXE or DLL */
APIRET APIENTRY RexxDeregisterFunction(const CHAR *name);
typedef APIRET APIENTRY RexxDeregisterFunctionPtr(const CHAR *);

/* To query if an External Function in an EXE or DLL is already registered */
APIRET APIENTRY RexxQueryFunction(const CHAR *name);
typedef APIRET APIENTRY RexxQueryFunctionPtr(const CHAR *);

/* Return Codes from the interpreter's External Function API (such as RexxRegisterSubcomExe().
 * These are also the values that a REXX script may get from REXXFUNCADD() or REXXFUNCDROP().
 * Also returned by RexxSetDllFunc/RexxSetExeFunc.
 */
#define RXFUNC_OK			0	/* Success */
#define RXFUNC_DEFINED		10	/* There is some function registered with this name */
#define RXFUNC_NOMEM		20	/* Not enough memory to perform the operation */
#define RXFUNC_NOTREG		30	/* There is no function registered with this name */
#define RXFUNC_MODNOTFND	40	/* The specified DLL can't be found */
#define RXFUNC_ENTNOTFND	50	/* The specified function can't be found in the DLL */
#define RXFUNC_NOTINIT		60	/* Registering a function in a DLL is not supported */
#define RXFUNC_BADTYPE		70	/* An arg you passed to the REXXSAA API is incorrect */
#define RXFUNC_BADVERSION	80	/* The DLL is not the desired version */

/* Reginald supports External Functions in Reginald-only DLLs. Such a DLL
 * must contain a callable function in it named RexxFeedback. This is the
 * declaration of it.
 */
typedef APIRET APIENTRY RexxFeedbackPtr(ULONG, void *);

typedef struct {
	void		*ReturnStr;
	const CHAR	*HelpBook;
	ULONG		Condition;
} REXXFUNCARGS;

/* The declaration for any External Functions you write specifically for Reginald. */
typedef APIRET APIENTRY RexxReginaldFunction(void *rexxPtr, void *params, REXXFUNCARGS *args);

/* Passed to RexxFeedback() */
#define RXREGGIE_DEREGFUNCS	0	/* DLL is being unloaded */
#define RXREGGIE_REGFUNCS	1	/* DLL is being loaded. Call RexxSetDllFunc() */
#define RXREGGIE_RAISE		2	/* A condition is being raised */
#define RXREGGIE_BREAK		3	/* Debugger has hit a breakpoint */
#define RXREGGIE_RESTART	4	/* Debugger is restarting from a breakpoint */
#define RXREGGIE_REINIT		5	/* Another script is using the DLL */

/* Passed to RexxSetDllFunc/RexxSetExeFunc */
#pragma pack(1)
struct REGINALDFUNC {
	void					*name;	// Pointer to the function's name (nul-terminated)
	RexxReginaldFunction	*func;	// Pointer to callable entry point
	unsigned char		required;	// Number of required args. -1 if we don't want call_function() to do the arg check
	unsigned char		numofargs;	// Max number of args passed to function
};
#pragma pack()

APIRET APIENTRY RexxSetDllFunc(void *rexxPtr, const UCHAR * libName, const UCHAR *rxName, struct REGINALDFUNC *funcAddrs, void *ptr);
typedef APIRET APIENTRY RexxSetDllFuncPtr(void *, const UCHAR *, const UCHAR *, struct REGINALDFUNC *, void *);

APIRET APIENTRY RexxSetExeFunc(const UCHAR *rxName, struct REGINALDFUNC *funcAddrs);
typedef APIRET APIENTRY RexxSetExeFuncPtr(const UCHAR *, struct REGINALDFUNC *);

APIRET APIENTRY RexxGetReginaldFuncs(void *block, ULONG blocksize, ULONG which);
typedef APIRET APIENTRY RexxGetReginaldFuncsPtr(void *, ULONG, ULONG);

#define REXXRET_VARS	0x00000001
#define REXXRET_MEM		0x00000002
#define REXXRET_FUNCS	0x00000004
#define REXXRET_ERRS	0x00000008
#define REXXRET_DEBUG	0x00000010
#define REXXRET_RUN		0x00000020
#define REXXRET_CALL	0x00000040
#define REXXRET_OBJ		0x00000080
#define REXXRET_ARGS	0x00000100

UCHAR * APIENTRY RexxNulTermArg(void *, void *, ULONG);
typedef UCHAR * APIENTRY RexxNulTermArgPtr(void *, void *, ULONG);

void * APIENTRY RexxParseFlags(void *rexxPtr, const UCHAR *buf, ULONG buflen, const UCHAR *strs, const unsigned long *values, unsigned long *dest, UCHAR **final);
typedef void * APIENTRY RexxParseFlagsPtr(void *, const UCHAR *, ULONG, const UCHAR *, const unsigned long *, unsigned long *, UCHAR **);
#define RXPARSE_FIRSTTIME 0x80000000
#define RXPARSE_ONLYONE 0x40000000
#define RXPARSE_NOERR 0x20000000
#define RXPARSE_GETARG 0x10000000

UCHAR * APIENTRY RexxGetArg(void *rexxPtr, void *parm, ULONG argnum, ULONG *length);
typedef UCHAR * APIENTRY RexxGetArgPtr(void *, void *, ULONG, ULONG *);

long APIENTRY RexxAsciiToNum(const UCHAR *val, unsigned long len, const UCHAR **next, ULONG flags);
typedef long APIENTRY RexxAsciiToNumPtr(const UCHAR *, unsigned long, const UCHAR **, ULONG);

UCHAR * APIENTRY RexxNumToAscii(ULONG num, UCHAR *buf, UCHAR flag);
typedef UCHAR * APIENTRY RexxNumToAsciiPtr(ULONG, UCHAR *, UCHAR);

#define REXXNUM_COMMA	0x0000002C /* Insert a comma between every 1000th place */
#define REXXNUM_UINT	0x80000000 /* Number is unsigned */
#define REXXNUM_INT		0x40000000 /* Number is signed */

ULONG APIENTRY RexxCountArgs(void *parm, ULONG flag);
typedef ULONG APIENTRY RexxCountArgsPtr(void *, ULONG);

typedef void * APIENTRY RexxApiErrMsgPtr(void *, UCHAR *, ULONG);





/* ====================== Conditions ====================== */

/* Return Codes from Asynchronous Request interface */
#define RXARI_OK				0
#define RXARI_NOT_FOUND			1
#define RXARI_PROCESSING_ERROR	2
#define RXARI_MEM				3

APIRET APIENTRY RexxSetHalt(LONG, LONG);
typedef APIRET APIENTRY RexxSetHaltPtr(LONG, LONG);

/* Reginald's proprietary API to directly to set some halt signal other
 * than SIGINT, or some condition besides HALT with appropriate error message
 */
APIRET APIENTRY RexxRaiseCondition(void *, ULONG, ULONG, const CHAR *, const CHAR *);
typedef APIRET APIENTRY RexxRaiseConditionPtr(void *, ULONG, ULONG, const CHAR *, const CHAR *);

/* Type of conditions that can be raised via RexxRaiseCondition() */
#define REXXRAISE_ERROR		0
#define REXXRAISE_FAILURE	1
#define REXXRAISE_HALT		2 
#define REXXRAISE_NOVALUE	3
#define REXXRAISE_NOTREADY	4
#define REXXRAISE_SYNTAX	5
#define NUMSTDCONDITIONS	6	/* Total # of standard REXX conditions that Reginald supports. User conditions are more than this */
#define REXXRAISE_USER		NUMSTDCONDITIONS






/* ================== EXTERNAL QUEUE INTERFACE =================== */

APIRET APIENTRY RexxCreateQueue(CHAR *, ULONG, const CHAR *, ULONG *);
typedef APIRET APIENTRY RexxCreateQueuePtr(CHAR *, ULONG, const CHAR *, ULONG *);

APIRET APIENTRY RexxDeleteQueue(const CHAR *);
typedef APIRET APIENTRY RexxDeleteQueuePtr(const CHAR *);

APIRET APIENTRY RexxQueryQueue(const CHAR *, ULONG *);
typedef APIRET APIENTRY RexxQueryQueuePtr(const CHAR *, ULONG *);

APIRET APIENTRY RexxAddQueue(const CHAR *, RXSTRING *, ULONG);
typedef APIRET APIENTRY RexxAddQueuePtr(const CHAR *, RXSTRING *, ULONG);

typedef struct
{
   USHORT hours;               /* hour of the day (24-hour) */
   USHORT minutes;             /* minute of the hour */
   USHORT seconds;             /* second of the minute */
   USHORT hundredths;          /* hundredths of a second */
   USHORT day;                 /* day of the month */
   USHORT month;               /* month of the year */
   USHORT year;                /* current year */
   USHORT weekday;             /* day of the week */
   ULONG  microseconds;        /* microseconds */
   ULONG  yearday;             /* day number within the year */
   USHORT valid;               /* valid time stamp marker */
} REXXDATETIME;
typedef REXXDATETIME *PDATETIME;

APIRET APIENTRY RexxPullQueue(const CHAR *, RXSTRING *, PDATETIME, ULONG);
typedef APIRET APIENTRY RexxPullQueuePtr(const CHAR *, RXSTRING *, PDATETIME, ULONG);

/* Request flags for External Data Queue access --------------------- */
#define RXQUEUE_FIFO	0	/* Access queue first-in-first-out */
#define RXQUEUE_LIFO	1	/* Access queue last-in-first-out  */

#define RXQUEUE_NOWAIT	0	/* Don't wait on an empty queue */
#define RXQUEUE_WAIT	1	/* Wait for data if queue empty */

/* Return Codes from RxQueue interface ------------------------------ */
#define RXQUEUE_OK		0		/* Successful return */
#define RXQUEUE_NOTINIT	1000	/* Queues not initialized */

#define RXQUEUE_STORAGE		1	/* Ret info buf not big enough	*/
#define RXQUEUE_SIZE		2	/* Data size > 64K-64			*/
#define RXQUEUE_DUP			3	/* Attempt-duplicate queue name	*/
#define RXQUEUE_NOEMEM		4	/* Not enough available memory	*/
#define RXQUEUE_BADQNAME	5	/* Not a valid queue name		*/
#define RXQUEUE_PRIORITY	6	/* Not accessed as LIFO|FIFO	*/
#define RXQUEUE_BADWAITFLAG	7	/* Not accessed as WAIT|NOWAIT	*/
#define RXQUEUE_EMPTY		8	/* No data in queue				*/
#define RXQUEUE_NOTREG		9	/* Queue does not exist			*/
#define RXQUEUE_ACCESS		10	/* Queue busy and wait active	*/
#define RXQUEUE_MAXREG		11	/* No memory to create a queue	*/
#define RXQUEUE_MEMFAIL		12	/* Failure in memory management	*/








/* ====================== Macros/Objects ======================== */

APIRET APIENTRY RexxAddMacro(
			const CHAR *,		/* Function to add */
			const CHAR *,		/* Name of file containing function */
			ULONG);				/* Flag indicating search pos */
typedef APIRET APIENTRY RexxAddMacroPtr(const CHAR *, const CHAR *, ULONG);

APIRET APIENTRY RexxAddMacroDll(
			CHAR *,		/* Macro DLL name */
			CHAR **);	/* Where to return pointer to full macro name */
typedef APIRET APIENTRY RexxAddMacroDllPtr(CHAR *, CHAR **);

APIRET APIENTRY RexxDropMacro(
			const CHAR *);		/* Name of function to remove */
typedef APIRET APIENTRY RexxDropMacroPtr(const CHAR *);

APIRET APIENTRY RexxSaveMacroSpace(
			ULONG,				/* Argument count (0==save all) */
			const CHAR **,		/* List of func names to save */
			const CHAR *);		/* File to save functions in */
typedef APIRET APIENTRY RexxSaveMacroSpacePtr(ULONG, const CHAR **, const CHAR *);

APIRET APIENTRY RexxLoadMacroSpace(
			ULONG,				/* Argument count (0==load all) */
			const CHAR **,		/* List of func names to load */
			const CHAR *);		/* File to load functions from */
typedef APIRET APIENTRY RexxLoadMacroSpacePtr(ULONG, const CHAR **, const CHAR *);

APIRET APIENTRY RexxQueryMacro(
			const CHAR *,		/* Function to search for */
			USHORT *);			/* Ptr for position flag return */
typedef APIRET APIENTRY RexxQueryMacroPtr(const CHAR *, USHORT *);

APIRET APIENTRY RexxReorderMacro(
			const CHAR *,		/* Name of func change order */
			ULONG);				/* New position for function */
typedef APIRET APIENTRY RexxReorderMacroPtr(const CHAR *, ULONG);

APIRET APIENTRY RexxClearMacroSpace(void);
typedef APIRET APIENTRY RexxClearMacroSpacePtr(void);

APIRET APIENTRY RexxCreateObject(void *rexxPtr, void * scriptName, void * objectVariableName, void ** objectHandle, void *index);
typedef APIRET APIENTRY RexxCreateObjectPtr(void *, void *, void *, void **, void *);

APIRET APIENTRY RexxGetObjectSource(void *rexxPtr, void * objectHandle, RAMSOURCE *line, ULONG lineno);
typedef APIRET APIENTRY RexxGetObjectSourcePtr(void *, void *, RAMSOURCE *, ULONG);

/* Registration Search Order Flags */
#define RXMACRO_SEARCH_BEFORE	1	/* Beginning of search order */
#define RXMACRO_SEARCH_AFTER	2	/* End of search order */
#define RXMACRO_RAM_SCRIPT		4	/* Reginald-only. The filename arg to RexxAddMacro() is really a buffer containing the script */
#define RXMACRO_PARENT			8	/* Reginald-only. Use the previous level to report raised conditions, and use its path if needed */
#define RXMACRO_LABELCHK		0x10	/* Reginald-only. Check for duplicate labels */


/* Return Codes from RxMacroSpace interface */
#define RXMACRO_OK					0	/* Macro operation successful */
#define RXMACRO_NO_STORAGE			1	/* Out of memory */
#define RXMACRO_NOT_FOUND			2	/* Requested macro not found */
#define RXMACRO_EXTENSION_REQUIRED	3	/* File ext required for save */
#define RXMACRO_ALREADY_EXISTS		4	/* Macro is already internally loaded */
#define RXMACRO_FILE_ERROR			5	/* File I/O error in save/load */
#define RXMACRO_SIGNATURE_ERROR		6	/* Incorrect format for load */
#define RXMACRO_SOURCE_NOT_FOUND	7	/* Requested macro source can't be found on disk */
#define RXMACRO_INVALID_POSITION	8	/* Invalid search order pos */
#define RXMACRO_NOT_INIT			9	/* API not initialized */

/* Return Codes from RexxCreateObject */
#define RXOBJECT_OK					0 /* Successful */	
#define RXOBJECT_OBJERR				1 /* Error creating the object, or object doesn't exist */
#define RXOBJECT_NOTRUNNING			2 /* No script currently running */







/* ====================== REXX CALLBACKS ======================== */

/* Proprietary Reginald/Regina API to call a subroutine/function in the currently running script */
APIRET APIENTRY RexxCallBack(const CHAR *,	/* Name of subroutine. Case-insensitive */
							 LONG,			/* Count of args passed */
							 RXSTRING *,	/* Array of args */
							 SHORT *,		/* Where to return a short from the subroutine, or 0 if not needed */
							 RXSTRING *);	/* Where to return the sub's return string, or 0 if not needed */
typedef APIRET APIENTRY RexxCallBackPtr(const UCHAR *, LONG, RXSTRING *, SHORT *, RXSTRING *);

/*
 * Return codes for RexxCallBack
 */
#define RX_CB_OK         0
#define RX_CB_BADP       1  /* Bad parameters */
#define RX_CB_NOTSTARTED 2  /* Interface not running */
#define RX_CB_TOOMANYP   3  /* Too many parameters */
#define RX_CB_MEMORY	 11	/* Not enough memory */
#define RX_CB_SCRIPT_ERROR	24	/* Interpretation error */
#define RX_CB_NOT_NUMERIC	26	/* Script did not return a whole number */
#define RX_CB_BADN       8  /* Procedure not found */
#define RX_CB_CONDITION	 (APIRET)-1	/* A condition was raised which the script needs to handle. You should not do anything to cause another condition to be raised (such as return anything except RXFUNC_OK from an external function) */

/* Queries if a subroutine/object exists in the current script, or in some created REXX object
 * and if so, fills in a special struct that can be referenced by RexxCallBack if you pass
 * a 0 for the first arg to RexxCallBack. Returns 0 if that subroutine/object doesn't exist
 */
void * APIENTRY RexxQuerySub(void *rexxPtr, const CHAR *subroutineName, ULONG flags, const CHAR *objectVarName);
typedef void * APIENTRY RexxQuerySubPtr(void *, const CHAR *, ULONG, const CHAR *);

APIRET APIENTRY RexxCallSub(void *rexxPtr, void *rexxQuerySubRet, ULONG argCount, RXSTRING *argList, RXSTRING *result);
typedef APIRET APIENTRY RexxCallSubPtr(void *, void *, ULONG, RXSTRING *, RXSTRING *);

#define RXNAME_CNT		0x20000000 /* name is not nul-terminated, and length is in low word of this same flags arg. */
#define RXNAME_NUL		0x40000000 /* name is nul-terminated */
#define RXNAME_LEN		0x80000000 /* name is not nul-terminated, and begins with a length byte. */

#define RXGETSUB_ISPTR	0x00400000 /* objectName is a special pointer. */
#define RXGETSUB_PARENT	0x00800000 /* Search parent levels for the sub. Not applicable to an object */
#define RXGETSUB_SYNTAX	0x01000000 /* Raise SYNTAX error if the sub doesn't exist */
/* #define RXGETSUB_USEPTR	0x02000000 */ /* Use special pointer last gotten upon previous call to RexxQuerySub */
#define RXGETSUB_GETPTR	0x04000000 /* Return special pointer to object. It can be used with RexxGetValue, RexxSetValue, RexxGetObjectSource */
#define RXGETSUB_SUBNUL	0x08000000 /* subroutineName is nul-terminated */
#define RXGETSUB_SUBLEN	0x10000000 /* subroutineName is not nul-terminated, and begins with a length byte. Otherwise must be nul-terminated */
#define RXGETSUB_OBJCNT	RXNAME_CNT /* name is not nul-terminated, and length is in low word of this same flags arg. */
#define RXGETSUB_OBJNUL	RXNAME_NUL /* objectVarName is nul-terminated */
#define RXGETSUB_OBJLEN	RXNAME_LEN /* objectVarName is not nul-terminated, and begins with a length byte. */

#ifdef __cplusplus
}
#endif

#endif /* __REXXSAA_H_INCLUDED */
