/* An example that registers two External Functions (ie, REXX-callable functions), TestAdd()
 * and TestDoCanadian() in a DLL. It also has "helper" functions that a REXX script can call to
 * easily register and deregister all of the External Functions in this DLL, TestAddFuncs()
 * and TestDropFuncs, respectively. Furthermore, it supports Reginald's auto-load feature
 * whereby the user can add this DLL to the REXX Administration Tool's list of auto-loaded
 * "Function Libraries" and all scripts will have immediate access to all of our
 * REXX callable functions without need to call TestAddFuncs(), nor TestDropFuncs().
 *
 * This also supports several Windows interpreters (besides Reginald).
 *
 * This example is for Microsoft Windows.
 */

#include <windows.h>
#include <stdio.h>
#define RX_STRONGTYPING
#include "..\rexxsaa.h"






/********************************************************************
 * The name of this DLL as far as the REXX script is concerned
 ********************************************************************/

const char LibName[] = "DLL4";

/********************************************************************
 * This is an array containing the names of all of the REXX callable
 * functions in this DLL. This is used when we initially register each
 * function with the REXX interpreter, and deregister upon closedown.
 ********************************************************************/

#define LONGEST_NAME	60 /* None of the names below exceed this length */

const char FncTable[] = {"TestAdd\0\
TestDoCanadian\0\
TestDropFuncs\0\
\0"};

/********************************************************************
 * The name of my Helper function to load all other External Functions
 ********************************************************************/

const char TestAddFuncsName[] = "TestAddFuncs";

/********************************************************************
 * Where we store the addresses of functions we call within the REXX
 * Interpreter DLL. The REXX interpreter is in DLL form itself. We dynamically
 * link to its functions (by fetching the addresses of its REXXSAA API
 * functions via WIN32's GetProcAddress() and stuffing them into the
 * following pointers to functions), rather than statically linking to
 * the interpreter's functions by telling the linker to include some
 * .LIB file for a specific interpreter. In this way, we support several
 * different WIN32 REXX interpreters.
 ********************************************************************/
RexxAllocateMemoryPtr		*RexxAllocateMemoryA;
RexxSetHaltPtr				*RexxSetHaltA;
RexxRegisterFunctionDllPtr	*RexxRegisterFunctionDllA;
RexxDeregisterFunctionPtr	*RexxDeregisterFunctionA;
RexxVariablePoolPtr			*RexxVariablePoolA;

/********************************************************************
 * Where we store the handle to our REXX Interpreter DLL
 ********************************************************************/
HMODULE		RexxHandle;

/********************************************************************
 * The names of some REXX Interpreter's we know about. (They must be
 * in DLL form). The ones we support (in this order) are Reginald,
 * Regina, Object Rexx, Enterprise, uni-REXX, and Querus.
 ********************************************************************/
const char InterpreterNames[] = {"REGINALD\0\
REGINA\0\
REXXAPI\0\
RXREXX\0\
RXX\0\
WREXX32\0\
\0"};	/* Must end with an extra 0 */

/********************************************************************
 * This is set to 1 if the interpreter we open is Reginald. Otherwise, 0
 ********************************************************************/
char		IsReginald;





/************************ deregister_funcs() *************************
 * Deregisters all of the functions in this DLL.
 *
 * stop =	The function name within FncTable[] at which to stop
 *			deregistering functions. Used in case of an error in
 *			register_funcs().
 */

void deregister_funcs(const char *stop)
{
    const char	*ptr;
	char		*dest;
	char		buf[LONGEST_NAME];

	/* Get first function name */
	ptr = &FncTable[0];

	/* Deregister each one of the REXX functions in this DLL */

	/* Another function? */
	while (*ptr && ptr < stop)
	{
		if (IsReginald)
		{
			(*RexxDeregisterFunctionA)(ptr);
			while (*(ptr)++);
		}
		else
		{
			/* Make the name upper-case */
			dest = &buf[0];
			do
			{
			} while ((*(dest)++ = toupper(*(ptr)++)));

			(*RexxDeregisterFunctionA)(&buf[0]);
		}
	}
}





/************************* register_funcs() **************************
 * Registers all of the functions in this DLL on behalf of the REXX
 * script.
 *
 * RETURNS: 0 if success, or one of the REXXSAA API error numbers for
 * RexxRegisterFunctionDll().
 */

int register_funcs(void)
{
	const char	*ptr;
	const char	*source;
	char		*dest;
	char		buf[LONGEST_NAME];
	int			rc;

	/* Get first function name */
	ptr = &FncTable[0];

	/* Register each one of the REXX functions in this DLL */

	/* Another function? */
	while (*ptr)
	{
		/* Reginald uppercases the name, so we don't need to do that */
		if (IsReginald)
		{
			if ((rc = (*RexxRegisterFunctionDllA)((char *)ptr, (char *)&LibName[0], (char *)ptr)))
			{
				/* An error. Deregister whatever functions we have
				 * already registered, and then get out of here.
				 */
				deregister_funcs(ptr);
				return(rc);
			}

			while (*(ptr)++);

			/* For other interpreters, we can't be sure if they
			 * uppercase the name, so we do it ourselves
			 */
		}
		else
		{
			/* Make the name upper-case */
			source = ptr;
			dest = &buf[0];
			do
			{
			} while ((*(dest)++ = toupper(*(ptr)++)));

			if ((rc = (*RexxRegisterFunctionDllA)(&buf[0], (char *)&LibName[0], (char *)source)))
			{
				deregister_funcs(source);
				return(rc);
			}

		}
	}

	return(0);
}





/***************************** TestAddFuncs() ******************************
 * Here's another of our REXX-callable functions. This one registers all of
 * the other REXX-callable functions in our DLL on behalf of the script. So,
 * script need only RXFUNCADD() this one, and call it once.
 *
 * Syntax:	err = TestAddFuncs()
 *
 * Params:	none
 *
 * RC Return: If success, an empty string. If error, one of the
 * error values from RexxRegisterFunctionDll().
 */

APIRET APIENTRY TestAddFuncs(char *name, unsigned long numargs, RXSTRING args[], char *queuename, RXSTRING *retstr)
{
	long	rc;

	/* Return empty string */
	retstr->strlength = 0;

	/* Register the functions */
	if ((rc = register_funcs()))
	{
		/* If an error, return the number from RexxRegisterFunctionDll() */
		sprintf(retstr->strptr, "%ld", rc);
		retstr->strlength = strlen(retstr->strptr);
	}

	/* Tell interpreter everything went OK */
	return(RXFUNC_OK);
}





/**************************** TestDropFuncs() *****************************
 * Here's another of our REXX-callable functions. This one deregisters all of
 * the other REXX-callable functions in our DLL on behalf of the script.
 *
 * Syntax:	CALL TestDropFuncs
 *			
 *			ADDRESS NULL
 *			TestDropFuncs()
 *
 * Params:	none
 *
 * RC Return: If success, an empty string.
 */

APIRET APIENTRY TestDropFuncs(char *name, unsigned long numargs, RXSTRING args[], char *queuename, RXSTRING *retstr)
{
	/* Deregister TestAddFuncs() on behalf of the script, It had to have
	 * registered it via RXFUNCADD() in order to call it. And
	 * deregister_funcs() does not deregister that one.
	 */
	if ((*RexxDeregisterFunctionA)((char *)&TestAddFuncsName[0]) == RXFUNC_OK)
	{
		/* Was TestAddFuncs() registered? If so, then the script
		 * must have RXFUNCADD()'ed it in order to call it. After all,
		 * this DLL does not auto-register TestAddFuncs(). In that case,
		 * we're going to honor its request to unload all functions.
		 * Otherwise, the script must have never called TestAddFuncs().
		 * Rather, this DLL's functions must have been auto-loaded by
		 * RexxRegisterFuncs(). In that case, we will ignore its attempt to
		 * deregister all of the functions now. Instead, Reginald will take
		 * care of deregistering our functions.
		 */

		/* Deregister all funcs in FncTable[] */
		deregister_funcs((const char *)-1);
	}
	
	/* Return an empty string */
	retstr->strlength = 0;

	/* Tell interpreter everything went OK */
	return(RXFUNC_OK);
}




/***************************** RexxRegisterFuncs() ******************************
 * This is called by the interpreter to auto-load our REXX-callable functions.
 * It must be called RexxRegisterFuncs().
 */

APIRET APIENTRY RexxRegisterFuncs(void)
{
	/* Register all of the REXX functions in this DLL */
	return(register_funcs());
}





/******************************** DLLMain() ********************************
 * Automatically called by Win32 when the DLL is loaded or unloaded. Here
 * is where we actually register and deregister our REXX-callable functions
 * to support Reginald's auto-load feature.
 *
 * For Microsoft Visual C++, you must call your DLL load/unload routine
 * DllMain(). For other compilers, they may require you to name it differently.
 * Consult the documentation with your compiler.
 */

BOOL WINAPI DllMain(HANDLE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
{
    switch(fdwReason)
	{
		/* ==============================================================
		 * Register all of the REXX functions in this DLL right as this
		 * DLL is being loaded. This enables Reginald's auto-load feature
		 * for this DLL.
		 */
		case DLL_PROCESS_ATTACH:
		{
			HMODULE						handle;
			const char					*ptr;

			/* Find out which REXX Interpreter loaded me. Assume Reginald
			 * (ie, that's the first one in our list). Make sure that it's
			 * one of our supported REXX interpreters and get a handle to
			 * it
			 */
			IsReginald = 1;
			ptr = &InterpreterNames[0];
			do
			{
				/* If this is it, stop searching */
				if ((handle = GetModuleHandle(ptr))) goto good;

				/* Can't be Reginald */
				IsReginald = 0;

				/* Next interpreter name */
				while (*(ptr)++);

				/* Any more? */
			} while (*ptr);

			/* I don't know about any others, so fail */
			return(0);

good:
			/* Get the address of the RexxVariablePool() function within the interpreter (ie,
			 * use dynamic linking to the interpreter)
			 */
			if (!(RexxVariablePoolA = (RexxVariablePoolPtr *)GetProcAddress((HINSTANCE)handle, "RexxVariablePool"))

				/* Get the address of the RexxSetHalt() function */
				|| !(RexxSetHaltA = (RexxSetHaltPtr *)GetProcAddress((HINSTANCE)handle, "RexxSetHalt"))

				/* Get the address of the RexxRegisterFunctionDll() function */
				|| !(RexxRegisterFunctionDllA = (RexxRegisterFunctionDllPtr *)GetProcAddress((HINSTANCE)handle, "RexxRegisterFunctionDll"))

				/* Get the address of the RexxDeregisterFunction() function */
				|| !(RexxDeregisterFunctionA = (RexxDeregisterFunctionPtr *)GetProcAddress((HINSTANCE)handle, "RexxDeregisterFunction")))
			{
				return(0);
			}

			/* Get the address of the RexxAllocateMemory() function. Some interpreters
			 * may not have this since it is non-standard
			 */
			RexxAllocateMemoryA = (RexxAllocateMemoryPtr *)GetProcAddress((HINSTANCE)handle, "RexxAllocateMemory");

			break;
		}

		/* ============================================================== */
		case DLL_THREAD_ATTACH:
		{
			/* We don't need to do anything for THREAD ATTACH, so we can
			 * disable this support.
			 */
			DisableThreadLibraryCalls(hinstDLL);
			break;
		}

/*
		case DLL_THREAD_DETACH:
			break;
*/
		/* ============================================================== */
/*
		case DLL_PROCESS_DETACH:
		{
		}
*/
	}

	/* Success */
	return(1);
}





/********************************* TestAdd() ***********************************
 * Here's one of our REXX-callable functions. This one expects two numeric args
 * from the script. It then adds the args and returns the sum to the script.
 *
 * Syntax:	sum = TestAdd(num1, num2)
 *
 * Params:	num1	=	First number
 *			num2	=	Second number
 *
 * RC Return: If success, the sum. If error, the SYNTAX condition is raised.
 */

APIRET APIENTRY TestAdd(CONST CHAR *name, ULONG numargs, RXSTRING args[], CONST UCHAR *queuename, RXSTRING *retstr)
{
	char	*stopptr;
	long	val1, val2;

	/* Check that there are 2, and only 2, args */
	if (numargs != 2 ||

		/* Make sure that both the args are there (ie, not omitted) */
		!args[0].strptr || !args[0].strlength || !args[1].strptr || !args[1].strlength)
	{
		/* Hey! He didn't supply at least 2 args. That's an error to me */

		/* One nice feature of Reginald is that, if you return RXERR_EXTFUNC_ERR
		 * (instead of simply RXERR_INCORRECT_CALL), you can copy an error
		 * message to 'retstr' and Reginald will report that to the script
		 * (or display the error message to the user).
		 *
		 * But even better, if you use one of the RXERR_ numbers that correspond
		 * to the ANSI GE numbers, or use the error numbers that utilize the
		 * MAKEFUNCARG() macro, you don't even need to supply an error message
		 * -- Reginald supplies one for you. Plus Reginald's online help can
		 * present a page on that specific error.
		 *
		 * This is a lot more informative than just reporting an "Incorrect call
		 * to function" as with other interpreters. But, only Reginald supports
		 * this. Although the following code should do nothing special on other
		 * properly written interpreters (ie, it will result in only that
		 * "Incorrect call to function" error message), it will allow Reginald
		 * to report more detailed errors for your function. Reginald will also
		 * report the name of our function, so we don't need to embed that in
		 * our error message.
		 */

		 /* Ok, did he omit both args? */
		if (!numargs || (!args[0].strptr && !args[1].strptr))
		{
			strcpy(retstr->strptr, "Arguments 1 and 2 can't be omitted");

			/* Note that we didn't have to nul-terminate the error message, but
			 * it was easier to do so above. Nevertheless, don't count that nul
			 */
			retstr->strlength = strlen(retstr->strptr);

			/* Tell Reginald to use the error message we're returning in 'retstr' */
			return(RXERR_EXTFUNC_ERR);

			/* Instead of doing the above, we could have simply done:
			 * return(RXERR_TOOFEWARGS | MAKEFUNCARG(2));
			 * But I wanted to show you how to return your own error message.
			 */
		}

		/* For the rest of these errors, we don't need to return a string.
		 * Reginald doesn't care about setting retstr to 0 if you return an
		 * error number. But other interpreters may
		 */
		if (!IsReginald)
		{
			retstr->strptr = 0;
			retstr->strlength = 0;
		}

		/* Did he pass in more than 2 args? */
		if (numargs > 2)
		{
			return(RXERR_TOOMANYARGS | MAKEFUNCARG(2));
		}
		 /* Ok, did he omit the first arg? */
		if (!args[0].strptr)
		{
			return(RXERR_ARGMISSING | MAKEFUNCARG(1));
		}
		 /* Ok, did he omit the second arg? */
		if (numargs < 2 || !args[1].strptr)
		{
			return(RXERR_ARGMISSING | MAKEFUNCARG(2));
		}
		 /* Ok, did he supply an empty string for the first arg? */
		if (!args[0].strlength)
		{
			return(RXERR_ARGNOTNUL | MAKEFUNCARG(1));
		}
		 /* Ok, did he supply an empty string for the second arg? */
		if (!args[1].strlength)
		{
			return(RXERR_ARGNOTNUL | MAKEFUNCARG(2));
		}

		/* No specific error message */
		return(RXERR_INCORRECT_CALL);
	}

	/* Remember that REXX passes numbers as numeric strings. We have
	   to convert each to a binary value in order to do the addition.
	   We also perform a check to make sure that the script passed in
	   numeric args (ie, there weren't any characters in the arg which
	   weren't valid in a numeric arg)
	 */
	val1 = strtol(args[0].strptr, &stopptr, 10);
	if ((ULONG)(stopptr - args[0].strptr) != args[0].strlength)
	{
		if (!IsReginald)
		{
			retstr->strptr = 0;
			retstr->strlength = 0;
		}
		return(RXERR_ARGWHOLENUM | MAKEFUNCARG(1));
	}

	val2 = strtol(args[1].strptr, &stopptr, 10);
	if ((ULONG)(stopptr - args[1].strptr) != args[1].strlength)
	{
		if (!IsReginald)
		{
			retstr->strptr = 0;
			retstr->strlength = 0;
		}
		return(RXERR_ARGWHOLENUM | MAKEFUNCARG(2));
	}

	/* Perform the addition */
	val2 += val1;

	/* Now we must return the sum as a numeric string to REXX */
	sprintf(retstr->strptr, "%d", val2);
	retstr->strlength = strlen(retstr->strptr);

	/* Tell interpreter everything went OK */
	return(RXFUNC_OK);
}





/****************************** TestDoCanadian() ********************************
 * Here's another of our REXX-callable functions. This one takes one arg, and
 * it appends the phrase ", eh?" to the end of it. It's a JOKE, people!
 *
 * Syntax:	new_phrase = TestDoCanadian(phrase)
 *
 * Params:	phrase	=	The original phrase
 *
 * RC Return: If success, the new phrase with ", eh?" appended. If error, the
 * SYNTAX condition is raised.
 */

char Canadian[] = {',',' ','e','h','?'};

APIRET APIENTRY TestDoCanadian(CONST CHAR *name, ULONG numargs, RXSTRING args[], CONST UCHAR *queuename, RXSTRING *retstr)
{
	/* Check that there is 1, and only 1, arg */
	if (numargs != 1 ||

		/* Make sure that the args is there (ie, not omitted) */
		!args[0].strptr)
	{
		/* Did he pass in more than 1 arg? */
		if (numargs > 1)
		{
			return(RXERR_TOOMANYARGS | MAKEFUNCARG(1));
		}

		 /* He must have omitted the arg */
		return(RXERR_ARGMISSING | MAKEFUNCARG(1));
	}
	
	/* If there's room, use retstr */
	if (retstr->strlength < args[0].strlength + sizeof(Canadian))
	{
		/* Not enough room. We need a new data buffer */

		/* See if our interpreter has RexxAllocateMemory(). If not, let's
		 * assume it wants us to use LocalAlloc(). Good luck.
		 */
		if (!RexxAllocateMemoryA)
		{
			if (!(retstr->strptr = LocalAlloc(LMEM_FIXED, args[0].strlength + sizeof(Canadian))))
				return(RXERR_STORAGE_EXHAUSTED);
		}
		else if (!(retstr->strptr = (*RexxAllocateMemoryA)(args[0].strlength + sizeof(Canadian))))
		{
			return(RXERR_STORAGE_EXHAUSTED);
		}
	}

	strcpy(retstr->strptr, args[0].strptr);
	memcpy(retstr->strptr + args[0].strlength, Canadian, sizeof(Canadian));

	retstr->strlength = args[0].strlength + sizeof(Canadian);

	/* Return successfully */
	return(RXFUNC_OK);
}
