/* An example that registers two External Functions (ie, REXX-callable functions), TestAdd()
 * and TestDoCanadian() in a DLL.
 */

#ifdef WIN32
#include <windows.h>
#endif
#include <stdio.h>
#define RX_STRONGTYPING
#include "..\rexxsaa.h"





/********************************* TestAdd() ***********************************
 * Here's one of our REXX-callable functions. This one expects two numeric args
 * from the script. It then adds the args and returns the sum to the script.
 *
 * Syntax:	sum = TestAdd(num1, num2)
 *
 * Params:	num1	=	First number
 *			num2	=	Second number
 *
 * RC Return: If success, the sum. If error, the SYNTAX condition is raised.
 */

APIRET APIENTRY TestAdd(CONST CHAR *name, ULONG numargs, RXSTRING args[], CONST UCHAR *queuename, RXSTRING *retstr)
{
	char	*stopptr;
	long	val1, val2;

	/* Check that there are 2, and only 2, args */
	if (numargs != 2 ||

		/* Make sure that both the args are there (ie, not omitted) */
		!args[0].strptr || !args[0].strlength || !args[1].strptr || !args[1].strlength)
	{
		/* Hey! He didn't supply at least 2 args. That's an error to me */

		/* One nice feature of Reginald is that, if you return RXERR_EXTFUNC_ERR
		 * (instead of simply RXERR_INCORRECT_CALL), you can copy an error
		 * message to 'retstr' and Reginald will report that to the script
		 * (or display the error message to the user).
		 *
		 * But even better, if you use one of the RXERR_ numbers that correspond
		 * to the ANSI GE numbers, or use the error numbers that utilize the
		 * MAKEFUNCARG() macro, you don't even need to supply an error message
		 * -- Reginald supplies one for you. Plus Reginald's online help can
		 * present a page on that specific error.
		 *
		 * This is a lot more informative than just reporting an "Incorrect call
		 * to function" as with other interpreters. But, only Reginald supports
		 * this. Although the following code should do nothing special on other
		 * properly written interpreters (ie, it will result in only that
		 * "Incorrect call to function" error message), it will allow Reginald
		 * to report more detailed errors for your function. Reginald will also
		 * report the name of our function, so we don't need to embed that in
		 * our error message.
		 */

		 /* Ok, did he omit both args? */
		if (!numargs || (!args[0].strptr && !args[1].strptr))
		{
			strcpy(retstr->strptr, "Arguments 1 and 2 can't be omitted");

			/* Note that we didn't have to nul-terminate the error message, but
			 * it was easier to do so above. Nevertheless, don't count that nul
			 */
			retstr->strlength = strlen(retstr->strptr);

			/* Tell Reginald to use the error message we're returning in 'retstr' */
			return(RXERR_EXTFUNC_ERR);

			/* Instead of doing the above, we could have simply done:
			 * return(RXERR_TOOFEWARGS | MAKEFUNCARG(2));
			 * But I wanted to show you how to return your own error message.
			 */
		}

		/* For the rest of these errors, we don't need to return a string.
		 * Reginald doesn't care about setting retstr to 0 if you return an
		 * error number. But other interpreters may
		 */
#ifndef USING_REGINALD
		retstr->strptr = 0;
		retstr->strlength = 0;
#endif

		/* Did he pass in more than 2 args? */
		if (numargs > 2)
		{
			return(RXERR_TOOMANYARGS | MAKEFUNCARG(2));
		}
		 /* Ok, did he omit the first arg? */
		if (!args[0].strptr)
		{
			return(RXERR_ARGMISSING | MAKEFUNCARG(1));
		}
		 /* Ok, did he omit the second arg? */
		if (numargs < 2 || !args[1].strptr)
		{
			return(RXERR_ARGMISSING | MAKEFUNCARG(2));
		}
		 /* Ok, did he supply an empty string for the first arg? */
		if (!args[0].strlength)
		{
			return(RXERR_ARGNOTNUL | MAKEFUNCARG(1));
		}
		 /* Ok, did he supply an empty string for the second arg? */
		if (!args[1].strlength)
		{
			return(RXERR_ARGNOTNUL | MAKEFUNCARG(2));
		}

		/* No specific error message */
		return(RXERR_INCORRECT_CALL);
	}

	/* Remember that REXX passes numbers as numeric strings. We have
	   to convert each to a binary value in order to do the addition.
	   We also perform a check to make sure that the script passed in
	   numeric args (ie, there weren't any characters in the arg which
	   weren't valid in a numeric arg)
	 */
	val1 = strtol(args[0].strptr, &stopptr, 10);
	if ((ULONG)(stopptr - args[0].strptr) != args[0].strlength)
	{
#ifndef USING_REGINALD
		retstr->strptr = 0;
		retstr->strlength = 0;
#endif
		return(RXERR_ARGWHOLENUM | MAKEFUNCARG(1));
	}

	val2 = strtol(args[1].strptr, &stopptr, 10);
	if ((ULONG)(stopptr - args[1].strptr) != args[1].strlength)
	{
#ifndef USING_REGINALD
		retstr->strptr = 0;
		retstr->strlength = 0;
#endif
		return(RXERR_ARGWHOLENUM | MAKEFUNCARG(2));
	}

	/* Perform the addition */
	val2 += val1;

	/* Now we must return the sum as a numeric string to REXX */
	sprintf(retstr->strptr, "%d", val2);
	retstr->strlength = strlen(retstr->strptr);

	/* Tell interpreter everything went OK */
	return(RXFUNC_OK);
}





/****************************** TestDoCanadian() ********************************
 * Here's another of our REXX-callable functions. This one takes one arg, and
 * it appends the phrase ", eh?" to the end of it. It's a JOKE, people!
 *
 * Syntax:	new_phrase = TestDoCanadian(phrase)
 *
 * Params:	phrase	=	The original phrase
 *
 * RC Return: If success, the new phrase with ", eh?" appended. If error, the
 * SYNTAX condition is raised.
 */

char Canadian[] = {',',' ','e','h','?'};

APIRET APIENTRY TestDoCanadian(CONST CHAR *name, ULONG numargs, RXSTRING args[], CONST UCHAR *queuename, RXSTRING *retstr)
{
	/* Check that there is 1, and only 1, arg */
	if (numargs != 1 ||

		/* Make sure that the args is there (ie, not omitted) */
		!args[0].strptr)
	{
		/* Did he pass in more than 1 arg? */
		if (numargs > 1)
		{
			return(RXERR_TOOMANYARGS | MAKEFUNCARG(1));
		}

		 /* He must have omitted the arg */
		return(RXERR_ARGMISSING | MAKEFUNCARG(1));
	}
	
	/* If there's room, use retstr */
	if (retstr->strlength < args[0].strlength + sizeof(Canadian))
	{
		/* Not enough room. We need a new data buffer */
		if (!(retstr->strptr = RexxAllocateMemory(args[0].strlength + sizeof(Canadian))))
		{
			return(RXERR_STORAGE_EXHAUSTED);
		}
	}

	strcpy(retstr->strptr, args[0].strptr);
	memcpy(retstr->strptr + args[0].strlength, Canadian, sizeof(Canadian));

	retstr->strlength = args[0].strlength + sizeof(Canadian);

	/* Return successfully */
	return(RXFUNC_OK);
}
